{
    This file is part of the chelinfo library.

    Copyright (c) 2008 by Anton Rzheshevski
    Parts (c) 2006 Thomas Schatzl, member of the FreePascal
    Development team
    Parts (c) 2000 Peter Vreman (adapted from original stabs line
    reader)

    Dwarf LineInfo Extractor

    See the file COPYING.FPC, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

 **********************************************************************}
{
    2008, Anton Rzheshevski aka Cheb:
    Like dr. Frankenshtein I sewn this library together
    from the dead meat of the the FPC RTL modules
    lineinfo.pp and lnfodwrf.pp.
    These (as of Jan. 2008 / FPC 2.2.0) both didn't work
    and had several limitations (e.g. inability to be used
    from a DLL)

    SUPPORTED TAGRETS: LINUX-32 AND WIN32 | FPC 2.2.0
    NOTE: Unlike the FPC RTL modules, this one does NOT
      have the "initialization" section: them buggers
      don't work in the Linux dlls.
      You must call the initialization function manually.

    }

{$mode delphi}
{$longstrings on}
{$codepage utf-8}
{$coperators on}

unit un_lineinfo;

interface

uses
  SysUtils, Classes, zstream;
  
  procedure GetLineInfo(addr: pointer; var exe, src: ansistring; var line, column: integer);
  {
    The format of returned information:
    "exe" *always* receives the full name of the executable file
      (the main exe or one of dlls it uses) the addr belongs to.
      In Linux, it returns the real file name, with all symlinks
      resolved.
    "line" can be negative, which means no line info has been found
      for this address. See LineInfoError (below) for details.
    "src" returns the source file name. It either doesn't or does
      contain a full path. If the source was in the same directory
      as the program itself, there will be no path. If the source
      was in the different directory, there will be a full path
      (for the moment when the program was compiled, NOT for the
      current location of that source).
    "column" is positive ONLY when there is more than one address
      stored for the same source line. FreePascal generates this
      on VERY rare occasions, mostly for the arithmetic formulas
      spanning several lines. So most of the time column will
      receive -1.
  }

  function InitLineInfo(someaddr: pointer): longbool;
  {
    This function is called by GetLineInfo() anyway if it doesnt't
      find a loaded line info for the executable to which the
      requested addres belongs.
    Also installs the custom BackTraceStr handler.

    Input:
    someaddr is adress of any function that belongs to the executable
      you want to pre-load the line info for. For example, a function
      exported from a particular dll.
    If you pass NIL, it will load the line info for the executable
      yhis module was compiled with.
      
    Output:
    Returns false if it failed to load the line info for the particular
      executable. In this case look LineInfoError for explanation
    Returns true if the line info for the particular executable is loaded ok.
    Returns true and does nothing if line info for that executable is
      already loaded.
  }
  
  procedure GetModuleByAddr(addr: pointer; var baseaddr: pointer; var filename: string);
  {
    This function allows you to know which executable (i.e. the main exe
      or one of the dlls loaded by it) owns this part of the virtual
      addres space.
    baseaddr receives the exe/dll base address
      (always NIL for the main exe in Linux).
    The mechnaism is made with possibility of a DLL relocation
      in mind, but that particular feature is untested.
    This function is used by GetLineInfo() to determine which executable
      to load line the info from.
  }
  
  var
    LineInfoError: WideString = '';
    LineInfoPaths: array of string = nil;
    {you can store the .zdli files in a different folder than the EXe itself.
      Just fill in this array.}

implementation

  uses
    {$ifdef unix}
      baseunix, dl
    {$else}
      windows
    {$endif}
    {$ifdef cge}
     ,{$ifdef cgemodule} mo_hub {$else} cge {$endif}
    {$endif}
    , un_xtrctdwrflnfo;

{$MACRO ON}

{define DEBUG_WRITE := WriteLn}
{$define DEBUG_WRITE := //}
{$ifdef cge}
  {define DEBUG_ADDLOG := AddLog}
  {$define DEBUG_ADDLOG := //}
{$else}
  {$define DEBUG_ADDLOG := //}
{$endif}

  function ChelinfoBackTraceStr(addr : Pointer) : ShortString;
  var
    exe, src: ansistring;
    line, column: integer;
    Store  : TBackTraceStrFunc;
  begin
    { reset to prevent infinite recursion if problems inside the code }
    Store := BackTraceStrFunc;
    BackTraceStrFunc := @SysBackTraceStr;
    GetLineInfo(addr, exe, src, line, column);
    { create string }
    Result:='  $' + HexStr(ptrint(addr), sizeof(ptrint) * 2);
    if line < 0 then Result+= '(no debug info: ' + LineInfoError + ')'
    else begin
      Result+= ', line ' + IntToStr(line);
      if column >=0 then Result+= ', column ' + IntToStr(column);
      Result += ' of ' + src;
    end;
    Result+= ' in ' + exe;
    BackTraceStrFunc := Store;
  end;




{$packrecords default}
  const
    MAX_RANDOM_OFFSET_TO_TRY = 1000;
  
  var
    initialized: boolean = false;

  { DWARF 2 default opcodes}
  const
    { Extended opcodes }
    DW_LNE_END_SEQUENCE = 1;
    DW_LNE_SET_ADDRESS = 2;
    DW_LNE_DEFINE_FILE = 3;
    { Standard opcodes }
    DW_LNS_COPY = 1;
    DW_LNS_ADVANCE_PC = 2;
    DW_LNS_ADVANCE_LINE = 3;
    DW_LNS_SET_FILE = 4;
    DW_LNS_SET_COLUMN = 5;
    DW_LNS_NEGATE_STMT = 6;
    DW_LNS_SET_BASIC_BLOCK = 7;
    DW_LNS_CONST_ADD_PC = 8;
    DW_LNS_FIXED_ADVANCE_PC = 9;
    DW_LNS_SET_PROLOGUE_END = 10;
    DW_LNS_SET_EPILOGUE_BEGIN = 11;
    DW_LNS_SET_ISA = 12;

  type
    { state record for the line info state machine }
    TMachineState = record
      address : cardinal;
      file_id : DWord;
      line : QWord;
      column : DWord;
      is_stmt : Boolean;
      basic_block : Boolean;
      end_sequence : Boolean;
      prolouge_end : Boolean;
      epilouge_begin : Boolean;
      isa : DWord;
      append_row : Boolean;
    end;


  { DWARF line number program header preceding the line number program, 32 bit version }
    TLineNumberProgramHeader32 = packed record
      unit_length : DWord;
      version : Word;
      length : DWord;
      minimum_instruction_length : Byte;
      default_is_stmt : byte;//Bool8;
      line_base : ShortInt;
      line_range : Byte;
      opcode_base : Byte;
    end;

    TDwarfChunk = packed record
      addr: pointer;
      line: integer;
      column, fileind: smallint; // it is sooo unlikely for them
    end;                         // to go beyond the 32767 limit...
    TFileInfo = packed record
      name: ansistring;
      dirind: integer;
    end;
    TDwarftable = array of TDwarfChunk;
    TCompilationUnit = record
      dTable: TDwarftable;
      Files: array of TFileInfo;
      Dirs: array of ansistring;
    end;
    TExecutableUnit = record
      name: string;
      CompilationUnit: array of TCompilationUnit;
    end;
  var
    base_addr: pointer = nil;
    ExecutableUnit: array of TExecutableUnit;
    

  procedure GetModuleByAddr(addr: pointer; var baseaddr: pointer; var filename: string);
  {$ifdef unix}
  var
    dlinfo: dl_info;
  begin
    FillChar(dlinfo, sizeof(dlinfo), 0);
    dladdr(addr, @dlinfo);
    baseaddr:= dlinfo.dli_fbase;
    filename:= String(dlinfo.dli_fname);
    if ExtractFileName(filename) = ExtractFileName(ParamStr(0))
      then baseaddr:= nil;
//    if filename = BacktrackSymlink(ParamStr(0)) then baseaddr:= nil; //doesn't work!
//addlog ('----------'#10#13'  %0'#10#13'  %1',[BacktrackSymlink(ParamStr(0)), filename]);
  end;
  {$else}
  var
    Tmm: TMemoryBasicInformation;
    TST: array[0..Max_Path] of Char;
  begin
    if VirtualQuery(addr, @Tmm, SizeOf(Tmm)) <> sizeof(Tmm)
      then raise Exception.Create('The VirualQuery() call failed.');
    baseaddr:=Tmm.AllocationBase;
    TST[0]:= #0;
    GetModuleFileName(THandle(Tmm.AllocationBase), TST, SizeOf(TST));
    filename:= String(PChar(@TST));
  end;
  {$endif}


  function InitLineInfo(someaddr: pointer): longbool;
  var
    dwarfsize: integer;
    dli, dc, ts: TStream;
    temp_length : DWord;
    unit_length, unit_base, next_base: dword;
    header_length: SizeInt;
    header : TLineNumberProgramHeader32;
    state : TMachineState;
    numoptable : array[1..255] of Byte;
    i, din: integer;
    c: dword;
    s: ansistring;
    b: byte;

    opcode, extended_opcode : Byte;
    extended_opcode_length : Integer;
    adjusted_opcode : Int64;
    addrIncrement, lineIncrement: Integer;
    _dwarf: pointer;
    ExeImageBase: cardinal;

    filename, exname: ansistring;

    { Reads an unsigned LEB encoded number from the input stream }
    function ReadULEB128() : QWord;
    var
      shift : Byte;
      data : Integer;
      val : QWord;
    begin
      shift := 0;
      result := 0;
      dli.Read (data, 1);
      while (data <> -1) do begin
        val := data and $7f;
        result := result or (val shl shift);
        inc(shift, 7);
        if ((data and $80) = 0) then
          break;
        dli.Read (data, 1);
      end;
    end;


    { Reads a signed LEB encoded number from the input stream }
    function ReadLEB128() : Int64;
    var
      shift : Byte;
      data : Integer;
      val : Int64;
    begin
      shift := 0;
      result := 0;
      dli.Read (data, 1);
      while (data <> -1) do begin
        val := data and $7f;
        result := result or (val shl shift);
        inc(shift, 7);
        if ((data and $80) = 0) then
          break;
        dli.Read (data, 1);
      end;
      { extend sign. Note that we can not use shl/shr since the latter does not
        translate to arithmetic shifting for signed types }
      result := (not ((result and (1 shl (shift-1)))-1)) or result;
    end;

    procedure SkipLEB128();
    var temp : int64;
    begin
      temp := ReadLEB128();
      DEBUG_ADDLOG('Skipping LEB128 : %0',[temp]);
    end;
    
    function CalculateAddressIncrement(_opcode : Byte) : Int64;
    begin
      result := (Int64(_opcode) - header.opcode_base) div header.line_range * header.minimum_instruction_length;
    end;

    
    function ReadString(): ansistring;
    var a: ansichar;
    begin
      Result:= '';
      while (true) do begin
        dli.Read(a, 1);
        if a = #0 then Exit;
        Result+= a;
      end;
    end;

    { initializes the line info state to the default values }
    procedure InitStateRegisters();
    begin
      with state do begin
        address := 0;
        file_id := 1;
        line := 1;
        column := 0;
        is_stmt := (header.default_is_stmt <> 0);
        basic_block := false;
        end_sequence := false;
        prolouge_end := false;
        epilouge_begin := false;
        isa := 0;
        append_row := false;
      end;
    end;
    
    function ParseCompilationUnit(var CompilationUnit: array of TCompilationUnit): boolean;
    var j: integer;
      procedure AddChunk;
      begin
        With CompilationUnit[high(CompilationUnit)] do begin
          SetLength(dtable, length(dtable) + 1);
          with dtable[high(dtable)] do begin
            //account for thepossible relocation (in 99% cases ExeImagebase = base_addr)
            addr:= state.address - ExeImagebase + base_addr;
            line:= state.line; // should we add 1 here ?
            column:= state.column;
            fileind:= state.file_id - 1;
            {$ifdef cge}
              if {$ifdef cgemodule} VerboseLog {$else} DebugMode {$endif}//  or true
                then
                  if (fileind < 0) or (fileind > high(files))
                    then AddLog('dwrflnfo: %0 %1:%2, INVALID fileind %3!', [addr, line, column, fileind])
                    else AddLog('dwrflnfo: %0 %1:%2 %3', [addr, line, column, files[fileind].name]);
            {$else}
              DEBUG_WRITE('dwrflnfo: ', IntToHex(cardinal(addr),8), ' ', line,' ', column, ' ',files[fileind].name);
            {$endif}
          end;
        end;
      end;


    begin
      Result:= true;
      fillchar(CompilationUnit[high(CompilationUnit)], sizeof(TCompilationUnit), 0);

     // a hack: the next compilation unit can have an unpredictable ofset,
     // so we try to find it by checking the most common values of the header
      j:= 0;
      repeat
        if unit_base + j + sizeof(header) + 2 > dli.Size then begin
          DEBUG_ADDLOG('The stream end reached, no more units.');
          Exit(false);
        end;
        dli.position:= unit_base + j;
        dli.Read(header, sizeof(header));
        with header do begin
          if (version = 2) and (line_range = 255) and (opcode_base = 13) then begin
            unit_base+= j;
            DEBUG_ADDLOG('rand_offset=%8, p=%9, unit_length %0 version %1  length %2  min_instr_leng %3 def_is_stmt %4 line_base %5 line_range %6 opcode_base %7',        [unit_length, version, length, minimum_instruction_length, default_is_stmt, line_base, line_range, opcode_base, j, pointer(unit_base)]);
            header_length := sizeof(header.length) + sizeof(header.version)
                          + sizeof(header.unit_length)
                          + header.length;
            Break;
          end;
        end;
        inc(j);
      until false;

      next_base:= unit_base + header.unit_length + sizeof(header.unit_length);

      fillchar(numoptable, sizeof(numoptable), #0);
      dli.Read(numoptable, header.opcode_base - 1);
      DEBUG_ADDLOG('Opcode parameter count table');
      for j := 1 to header.opcode_base - 1 do begin
        DEBUG_ADDLOG('Opcode[%0] - %1 parameters', [j, numoptable[j]]);
      end;

      With CompilationUnit[high(CompilationUnit)] do begin
        SetLength(dirs, 1);
        dirs[0]:=''; //the project directory
        while (true) do begin
          s:= ReadString();
          if (s = '') then break;
          SetLength(dirs, length(dirs) + 1);
          dirs[high(dirs)]:= s;
          DEBUG_ADDLOG('Dir %0: %1',[high(dirs), AnsiToWide(s)]);
        end;


        while (true) do begin
          s:= ReadString;
          if (s = '') then break;
          SetLength(files, length(files) + 1);
          with files[high(files)] do begin
            name:= s;
            dirind:= ReadLEB128(); { the directory index for the file }
            DEBUG_ADDLOG('File %0 (dir %2): %1',[high(files), AnsiToWide(name), dirind]);
          end;
          SkipLEB128(); { skip last modification time for file }
          SkipLEB128(); { skip length of file }
        end;


        dli.Position:= header_length + unit_base;
        unit_length:= header.unit_length;
        
        InitStateRegisters();

        while (dli.Position - unit_base) < unit_length - 2{ dli.Size - 1} do begin
          dli.Read(opcode, 1);
          DEBUG_ADDLOG('Next opcode: %0  (stream pos. %1 ( %2 / %3 )',[opcode, dli.position, dli.position - unit_base, unit_length]);

          case (opcode) of
            { extended opcode }
            0 : begin
              extended_opcode_length := ReadULEB128();
              dli.Read(extended_opcode, 1);
              case (extended_opcode) of
                DW_LNE_END_SEQUENCE : begin
                  state.end_sequence := true;
                  state.append_row := true;
                  AddChunk;
                  DEBUG_ADDLOG('DW_LNE_END_SEQUENCE');
                  InitStateRegisters();
                end;
                DW_LNE_SET_ADDRESS : begin
                  dli.Read(state.address, 4);
                  DEBUG_ADDLOG('DW_LNE_SET_ADDRESS (%0)', [pointer(state.address)]);
                end;
                DW_LNE_DEFINE_FILE : begin
                  {$ifdef DEBUG_DWARF_PARSER}s := {$endif}ReadString();
                  SkipLEB128();
                  SkipLEB128();
                  SkipLEB128();
                  DEBUG_ADDLOG('DW_LNE_DEFINE_FILE (' + s + ')');
                end;
                else begin
                  DEBUG_ADDLOG('Unknown extended opcode (opcode %0 length %1)', [extended_opcode, extended_opcode_length]);
                  dli.Position:= dli.Position + extended_opcode_length - 1;
                end;
              end;
            end;
            DW_LNS_COPY : begin
              state.basic_block := false;
              state.prolouge_end := false;
              state.epilouge_begin := false;
              state.append_row := true;
              DEBUG_ADDLOG('DW_LNS_COPY');
              AddChunk;
            end;
            DW_LNS_ADVANCE_PC : begin
              inc(state.address, ReadULEB128() * header.minimum_instruction_length);
              DEBUG_ADDLOG('DW_LNS_ADVANCE_PC (' + hexstr(state.address, sizeof(state.address)*2) + ')');
            end;
            DW_LNS_ADVANCE_LINE : begin
              inc(state.line, ReadLEB128());
              DEBUG_ADDLOG('DW_LNS_ADVANCE_LINE (%0)', [state.line]);
             // AddChunk;
            end;
            DW_LNS_SET_FILE : begin
              state.file_id := ReadULEB128();
              DEBUG_ADDLOG('DW_LNS_SET_FILE (%0)', [state.file_id]);
            end;
            DW_LNS_SET_COLUMN : begin
              state.column := ReadULEB128();
              DEBUG_ADDLOG('DW_LNS_SET_COLUMN (%0)', [state.column]);
            end;
            DW_LNS_NEGATE_STMT : begin
              state.is_stmt := not state.is_stmt;
              DEBUG_ADDLOG('DW_LNS_NEGATE_STMT (%0)',[state.is_stmt]);
            end;
            DW_LNS_SET_BASIC_BLOCK : begin
              state.basic_block := true;
              DEBUG_ADDLOG('DW_LNS_SET_BASIC_BLOCK');
            end;
            DW_LNS_CONST_ADD_PC : begin
              inc(state.address, CalculateAddressIncrement(255));
              DEBUG_ADDLOG('DW_LNS_CONST_ADD_PC (' + hexstr(state.address, sizeof(state.address)*2) + ')');
            end;
            DW_LNS_FIXED_ADVANCE_PC : begin
              dli.Read(state.address, sizeof(state.address));
              DEBUG_ADDLOG('DW_LNS_FIXED_ADVANCE_PC (' + hexstr(state.address, sizeof(state.address)*2) + ')');
            end;
            DW_LNS_SET_PROLOGUE_END : begin
              state.prolouge_end := true;
              DEBUG_ADDLOG('DW_LNS_SET_PROLOGUE_END');
            end;
            DW_LNS_SET_EPILOGUE_BEGIN : begin
              state.epilouge_begin := true;
              DEBUG_ADDLOG('DW_LNS_SET_EPILOGUE_BEGIN');
            end;
            DW_LNS_SET_ISA : begin
              state.isa := ReadULEB128();
              DEBUG_ADDLOG('DW_LNS_SET_ISA (%0)', [state.isa]);
            end;
            else begin { special opcode }
              if (opcode < header.opcode_base) then begin
                DEBUG_ADDLOG('Unknown standard opcode $' + hexstr(opcode, 2) + '; skipping');
                for j := 1 to numoptable[opcode] do
                  SkipLEB128();
              end else begin
                adjusted_opcode := integer(opcode) - header.opcode_base;
                addrIncrement := CalculateAddressIncrement(opcode);
                inc(state.address, addrIncrement);
                lineIncrement := header.line_base + (adjusted_opcode mod header.line_range);
                inc(state.line, lineIncrement);
                DEBUG_ADDLOG('Special opcode $' + hexstr(opcode, 2) + ' address increment: %0 new line: %1', [addrIncrement, lineIncrement]);
                state.basic_block := false;
                state.prolouge_end := false;
                state.epilouge_begin := false;
                state.append_row := true;
                AddChunk;
              end;
            end;
          end; //case
        end; //while
      end; //with
      Result:= true;
    end;
    
    
  begin
    if someaddr = nil then someaddr:=@InitLineInfo;
    GetModuleByAddr(someaddr, base_addr, filename);
    din:= -1;
    for i:=0 to high(ExecutableUnit) do
      if ExecutableUnit[i].name = filename then din:=i;
    if din < 0 then begin
      SetLength(ExecutableUnit, length(ExecutableUnit) + 1);
      din:= high(ExecutableUnit); // it gets added at the end, of course
      ExecutableUnit[din].name:= filename;
      ExecutableUnit[din].CompilationUnit:= nil;
    end
    else
      Exit(true); //already initialized for this exe/dll

    BackTraceStrFunc := @ChelinfoBacktraceStr;

    {$ifdef cge}
    AddLog(RuEn('Загрузка информации для самоотладки...', 'Loading the self-debugging info...'));
    if {$ifdef cgemodule} VerboseLog {$else} DebugMode {$endif} or true
      then AddLog('  %2 is %1, base %0', [base_addr, filename, someaddr]);
    {$else}
//writeln('-- ',IntToHex(cardinal(base_addr), 8), '  ',filename);
    {$endif}

    try
      Result:= false;
     {  First, try the external file with line information.
        Failing that, try to parse the executable itself }

      exname:= DlnNameByExename(filename);
      i:= -1;
      repeat
//{$ifdef cge}addlog('%0  %1', [exname, FileExists(exname)]);{$endif}
        if FileExists(exname)
          then break
          else exname:='';
        inc(i);
        if i > high(LineInfoPaths) then break;
        exname:= LineInfopaths[i] + DlnNameByExename(ExtractFileName(filename));
      until false;
      
      if exname <> ''
        //and (FileAge(filename) <= FileAge(DlnNameByExename(filename)))
      then begin
        //the compression streams are unable to seek,
        //so we decompress to a memory stream first.
        ts:=TFileStream.Create(exname, fmOpenRead);
        dc:=TDecompressionStream.Create(ts);
        dli:= TMemoryStream.Create;
        dc.Read(DwarfSize, 4);
        dc.Read(ExeImageBase, 4);
        dli.CopyFrom(dc, DwarfSize);
        dc.Free;
        ts.Free;
      end
      else begin
        DEBUG_WRITE('  External file not found, trying to parse self.');
        if not ExtractDwarfLineInfo(filename, _dwarf, DwarfSize, ExeImageBase) then begin
          LineInfoError:= ExtractDwarfLineInfoError;
          Exit(false);
        end;
        dli:= TMemoryStream.Create;
        dli.Write(_dwarf^, DwarfSize);
        FreeMem(_dwarf);
      end;

      dli.Position:= 0;
      next_base:= 0;
      
      DEBUG_WRITE('dwarf line info: ',dli.size,' bytes.');
      
      With ExecutableUnit[din] do
        while true {next_base < dli.Size - sizeof(header) - 5} do begin
          SetLength(CompilationUnit, length(CompilationUnit) + 1);
          unit_base:= next_base;
          if not ParseCompilationUnit(CompilationUnit) then begin
            SetLength(CompilationUnit, length(CompilationUnit) - 1);
            Break;
          end;
        end;
    except
      LineInfoError:=
        {$ifdef cge}RuEn('Крах при парсинге отладочной информации: ',{$endif}
        'Crashed parsing the dwarf line info: '{$ifdef cge}){$endif}
        + (ExceptObject as Exception).Message;
      dli.Free;
      Result:=false;
    end;
    if {Result and} (length(ExecutableUnit[din].CompilationUnit) > 0)
      then begin
        initialized:= true;
        {$ifdef cge}
        AddLog(RuEn('  найдено %0 блоков.','  found %0 units.'), [length(ExecutableUnit[din].CompilationUnit)])
        {$else}
        DEBUG_WRITE('  found ', length(ExecutableUnit[din].CompilationUnit), ' compilation units.');
        {$endif}
      end
      else begin
        {$ifdef cge}
        AddLog(RuEn('  не найдено ни одного блока для %0.','  not found any units in %0.'), [ExecutableUnit[din].name]);
        {$endif}
        LineInfoError:= 'no compilation units found.';
        Result:=false;
      end;
    dli.Free;
    Result:= True;
  end;
  



  procedure GetLineInfo(addr: pointer; var exe, src: ansistring; var line, column: integer);
  var
    i,j,k, ei: integer;
    ubase: pointer;
    b: boolean;
  begin
    src:='';
    exe:='';
    line:= -1;
    column:= -1;
    //LineInfoError:= '';

    if not initialized then
      if not InitLineInfo(addr) then exit;
    try
      GetModuleByAddr(addr, ubase, exe);
      ei:= -1;
      for i:=0 to high(ExecutableUnit) do
        if ExecutableUnit[i].name = exe then ei:=i;
      if ei < 0 then begin
        if not InitLineInfo(addr) then exit;
        ei:= high(ExecutableUnit); // it gets added at the end, of course
      end;
      with ExecutableUnit[ei] do
        for j:=0 to high(CompilationUnit) do
          with ExecutableUnit[ei].CompilationUnit[j] do
            for i:=0 to high(dtable) do
              if (addr = dtable[i].addr)
                or ((i < high(dtable)) and (addr >= dtable[i].addr) and (addr < dtable[i + 1].addr))
              then begin
                src:= Dirs[Files[dtable[i].fileind].dirind] + Files[dtable[i].fileind].name;
                line:= dtable[i].line;
                column:= dtable[i].column;
                //now check if the same line appears twice with different columns.
                // if not, then reset column to -1.
                b:= false;
                for k:=0 to high(dtable) do
                  b:= b or
                      (     (dtable[k].line = line)
                        and (dtable[k].column <> column)
                        and (dtable[k].fileind = dtable[i].fileind));
                if not b then column:= -1;
                LineInfoError:= '';
                Exit;
              end;
    except
      LineInfoError:= (ExceptObject as Exception).Message;
    end;
  end;
  
end.

