{
    This file is part of the Automaton program,
    Copyright (c) 2004 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the *.manga command file parser.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}
 
{$mode delphi}
Unit aut_parser;

Interface

Uses
  SysUtils, Classes, cl_typedefs, cl_dyna, cl_classes,
  cl_strings, aut_graph, cl_kambipng, IniFiles;

Const
  VersionString = '0.98, beta';
  MinimalFontSize = 5;
  MaximalFontSize = 100;
  MinBorderDiameter = -10;
  MaxBorderDiameter = 30;

Type
  TCommand =
    (CBegin, CEnd, CNamePrefix, CFont, CFontSize, CMinFontSize,
     CMaxFontCompression, CSFX, CCleaningPower, CTexture, CPaletteDepth,
     CDisablePNG, CMinWidth, CMaxWidth, CMinHeight, CMaxHeight,
     CDontScale, CScaleBefore, CScaleAfter, CHardcoremania,
     CColor, CBlackAndWhite, CStdBalons, CStdNegaBalons, CBalonsBgr,
     CTransparentBalonsBgr, CBalonsPars, CLineSpacing, CDisableAutoHypens,
     CAutoHypens, CCompressionLoathing, CShortLinesLoathing, CTransitionsLoathing,
     CDeviationsLoathing, CSmearMetod, CSmearPower, CSmearInBalons,
     CDontSmearInBalons, CDraft, CKillPatterns, CDontKillPatterns,
     CReliz, CAltBalon, CBalon,
     NotACommand);
Const
  CommandsStr: array [TCommand] of string =
    ('','','__','','_',
     '__', '__' ,'',
     '_','','_', '_',
     '_','_','_',
     '_','___',
     '__','__',
     '__', '_', 'ר-_',
     '_', '__', '_',
     '_', '_', '_',
     '_','____',
     '__','__ ',
     '__','__','_',
     '_','__','___',
     '','_','__','','','','');
   SmearMethodname: array [TSmear] of string =
    ('', '', 'ƨ', '',
     '', '', '','');//'', '');
    
    CommandsAllowedInPage: set of TCommand =
      [CFontSize, CMaxFontCompression, CSFX, CCleaningPower, CBalon,
       CPaletteDepth, CMinWidth, CMaxWidth, CMinHeight, CMaxHeight,
       CScaleBefore, CScaleAfter, CDontScale,  CColor, CBlackAndWhite,
       CStdBalons, CBalonsBgr, CTransparentBalonsBgr, CBalonsPars,
       CLineSpacing, CDisableAutoHypens, CAutoHypens, CCompressionLoathing,
       CShortLinesLoathing, CTransitionsLoathing, CSmearMetod, CSmearPower,
       CDeviationsLoathing, CSmearInBalons, CDontSmearInBalons, CKillPatterns,
       CDontKillPatterns, CAltBalon];
    CommandsAllowedBetweenBalons: set of TCommand = [
       CFontSize, CBalon, CStdBalons, CBalonsBgr, CTransparentBalonsBgr,
       CBalonsPars, CSFX, CLineSpacing, CSmearInBalons, CDontSmearInBalons,
       CAltBalon];
    CommandsAllowedGlobally: set of TCommand =
      [CBegin, CEnd, CNamePrefix, CFont, CFontSize, CMaxFontCompression,
       CCleaningPower, CTexture, CPaletteDepth, CDisablePNG, CMinWidth,
       CMaxWidth, CMinHeight, CMaxHeight, CScaleBefore, CScaleAfter,
       CDontScale, CHardcoremania, CColor, CBlackAndWhite, CStdBalons,
       CBalonsBgr, CTransparentBalonsBgr, CBalonsPars, CLineSpacing,
       CDisableAutoHypens, CAutoHypens, CCompressionLoathing, CMinFontSize,
       CShortLinesLoathing, CTransitionsLoathing, CDeviationsLoathing,
       CSmearMetod, CSmearPower, CSmearInBalons, CDontSmearInBalons,
       CDraft, CKillPatterns, CDontKillPatterns, CReliz];
    CommandsNotAllowedBetweenPages: set of TCommand =
      [CFont, CDisablePng, CHardcoreMania, CDraft, CReliz];
    Draft = - 1;
    Reliz = 1;

Type
  TParserHelper = class(TStringListWithIDs)
  private
    _BaseDir: string;
    CurrentLine,
    CurrentPos: integer;
    MinErrLine, MaxErrLine: integer;
    GlobalLevel: boolean;
    function ExtractEntryName: string;
    function ExtractCommandPars: TStringListWithIDs;
    procedure ComplainForError(expected, whatwegot, suggestion: string);
    procedure ComplainForSyntaxError(s: string);
    procedure ComplainForUnknownCommand(c: string);
    procedure ComplainForGenericError(s: string);
    function StrToCommand(s: string): TCommand;
    function StrToSmear(s: string): TSmear;
    function ErrorLocation: string;
    function ParNameToCanonicForm(pm: string): string;
  end;

  TParamsContainer = class(TParserHelper)
  private
    CleaningPower,
    FontSize, MinFontSize, FontCompression, PaletteDepth,
    MinWidth,MaxWidth, MinHeight,MaxHeight, BalonColor: integer;
    ScaleBeforeProcessing, InColor: LongBool;
    FileNamePrefix, BalonsPars, BalonSmear: string;
    SFX, Balon, BalonPrs, BalonsSmear: TStringList;
    BalonBgr, BalonFontSize, BalonSpacing: TAOI;
    HypenLen: integer;
    CompressionLoathing,  ShortLinesLoathing,
    TransitionsLoathing, DeviationsLoathing: float;
    LineSpacing: float;
    SmearMethod: TSmear;
    SmearPower, PattToKill: Integer;
    HasSmearInbalons: integer;
    AlreadyApplied: set of TCommand;
    function ParseTriplet(s: string;  ersi: integer): string; VIRTUAL; ABSTRACT;
    function ParseSFX(s: string; IsSFX: boolean): string; VIRTUAL; ABSTRACT;
    function ParseBalon(s: string): string; VIRTUAL; ABSTRACT;
  public
    Constructor Create;
    Destructor Destroy; Override;
    Procedure ApplyCommand(Command: TCommand; Params: TStringListWithIDs);
    function ColorToIntVerbose(str: string; TransparentAllowed: boolean): integer;
    function StrToIntVerbose(str: string; Min, Max: integer): integer;
    function StrToFloatVerbose(str: string; Min, Max: float): float;
    function BitDpthToIntVerbose(str: string): integer;
    function StrToIntValid(str: string; Min, Max: integer): boolean;
    function StrToFloatValid(str: string; Min, Max: float): boolean;
    procedure ClearSizeLimits;
    function ColorTextureCheck(t: string): string; VIRTUAL; ABSTRACT;
  end;

  TAuManga = class(TParamsContainer)
  private
    Dir, OutputDir, _filename: string;
    TextureName, Texture, Page, PagePars: TAOS;
    Font: TArrayOfFont;
    Summaries: TMemIniFile;
    Hardcore: boolean;
    Pages: TAOO;
    GMode: Integer;
  public
    Constructor Create(FileName: string);
    Destructor Destroy; OVERRIDE;
    function ColorTextureCheck(t: string): string; OVERRIDE;
    Procedure Parse;
  end;

  TAuMangaPage = class(TParamsContainer)
  private
    Scan: TAuMangaScan;
    FileName: string;
    Parent: TAuManga;
    function ParseTriplet(s: string;  ersi: integer): string; OVERRIDE;
    function ParseSFX(s: string; IsSFX: boolean): string; OVERRIDE;
    function ParseBalon(s: string): string; OVERRIDE;
  public
    Constructor Create(_Parent: TAuManga; Contents: TStringListWithIDs; Name, Dir: String);
    Destructor Destroy; OVERRIDE;
    function Summary: string;
    function SmearSummary: string;
    function ColorTextureCheck(t: string): string; OVERRIDE;
    procedure Parse;
    procedure Execute;
  end;


  Function UpperCase(s: string):string;
  function UpCase(h: char): char;

//                        ,-- %< >--.
//  : >--+-->--+---- <>------+--(<>)--+--->
//                  '---------------<----------------------------'

Implementation

  Function GetFileDateWithExtMs(name: string): string;
  var i, j, sum: integer;
  begin
    sum:=0;
    name:=ChangeFileExt(name, '');
    j:=0;
    Repeat
      i:=FileAge(ChangeFileExt(name, '.png')) + FileAge(ChangeFileExt(name, '.tga'));
      sum:=sum + i;
      name:=name + '_';
    Until i < 0;
    Result:=IntToStr(sum);
  end;
  
  Function GetFileOutDate(name: string): string;
  begin
    Result:=IntToStr(FileAge(ChangeFileExt(ChangeFileExt(name, ''), '.png'))
                   + FileAge(ChangeFileExt(ChangeFileExt(name, ''), '.bmp')));
  end;

  function TAOIToStr(A: TAOI):string;
  var j: integer;
  begin
    Result:='';
    For j:=0 to A.High do Result:=Result + IntToStr(A[j]);
  end;

  function BoolToStr(b: boolean): string;
  begin
    if b then Result:='1' else Result:='0';
  end;
  
  function TAuMangaPage.Summary: string;
  var i, j: integer;
  begin
    Result:=GetFileDateWithExtMs(FileName)
     + format('%s%d%d%d%d%d%d%d%d%d',
       [ FileNamePrefix, CleaningPower,
        PaletteDepth, MinWidth, MinHeight, MaxWidth, MaxHeight,
        Integer(ScaleBeforeProcessing), Integer(InColor), PattToKill])
    + StrDelSet(SFX.Text, [#10, #13]) + StrDelSet(BalonsSmear.Text,[#10,#13]);
    if Balon.Count > 0 then begin
      Result:=Result + StrDelSet(Balon.Text + BalonPrs.Text, [#10, #13])
      + format('%d%d%d%d%f%f%f%f',[HypenLen, FontSize, MinFontSize, FontCompression,
          CompressionLoathing, ShortLinesLoathing, TransitionsLoathing, DeviationsLoathing])
      + BoolToStr(Parent.Hardcore or (Pos(TagHardcore, Balon.Text) > 0))
      + TAOIToStr(BalonBgr) + TAOIToStr(BalonFontSize) + TAOIToStr(BalonSpacing);
      For j:=0 to 9 do if Assigned(Parent.Font[j])
        then Result:=Result + Parent.Font[j].Name;
      if SmearPower = 0 then Result:=Result + StrDelSet(BalonsSmear.Text, [#10, #13]);
    end;
  end;
  
  function TAuMangapage.SmearSummary: string;
  begin
    Result:=Trim(format('%s %d %s', [SmearMethodName[SmearMethod], SmearPower, StrDelSet(BalonsSmear.Text, [#10, #13])]));
  end;

  function TAuManga.ColorTextureCheck(t: string): string;
  var j: integer;
  begin
    Result:=t;
    t:=ParNameToCanonicForm(t);
    For j:=0 to Texture.High do
      if t = TextureName[j] then Result:=Texture[j];
  end;

  function TAuMangaPage.ColorTextureCheck(t: string): string;
  var j: integer;
  begin
    Result:=t;
    t:=ParNameToCanonicForm(t);
    For j:=0 to Parent.Texture.High do
      if t = Parent.TextureName[j] then Result:=Parent.Texture[j];
  end;
  
  Constructor  TAuMangaPage.Create(_Parent: TAuManga; Contents: TStringListWithIDs; Name, Dir: String);
  var j: integer;
  begin
    inherited Create;
    Parent:=_Parent;
    For j:=0 to Contents.Count - 1 do begin
      Add(Contents[j]);
      Id[j]:=Contents.Id[j];
    end;
    Contents.Free;
    GlobalLevel:=False;
    CleaningPower:=Parent.CleaningPower;
    FontSize:=Parent.FontSize;
    MinFontSize:=Parent.MinFontSize;
    FontCompression:=Parent.FontCompression;
    PaletteDepth:=Parent.PaletteDepth;
    MinWidth:=Parent.MinWidth;
    MaxWidth:=Parent.MaxWidth;
    MinHeight:=Parent.MinHeight;
    MaxHeight:=Parent.MaxHeight;
    BalonColor:=Parent.BalonColor;
    BalonsPars:=Parent.Balonspars;
    BalonSmear:=Parent.BalonSmear;
    ScaleBeforeProcessing:=Parent.ScaleBeforeProcessing;
    InColor:=Parent.InColor;
    LineSpacing:=parent.LineSpacing;
    HypenLen:=Parent.HypenLen;
    PattToKill:=Parent.PattToKill;
    CompressionLoathing:=Parent.CompressionLoathing;
    ShortLinesLoathing:=Parent.ShortLinesLoathing;
    TransitionsLoathing:=Parent.TransitionsLoathing;
    DeviationsLoathing:=Parent.DeviationsLoathing;
    SmearMethod:= Parent.SmearMethod;
    SmearPower:= Parent.SmearPower;
    FileName:=Dir + Name;
  end;

  Destructor TAuMangaPage.Destroy;
  begin
    SafeFree(Scan);
    inherited;
  end;




  Const
    ers1: array[0..2] of string = (' ','   ',' ');
    ers2: array[0..2] of string = ('"()"','"_()"','"__()"');

  function TAuMangaPage.ParseSFX(s: string;  IsSFX: boolean): string;
  begin
    if IsSfx then Result:=ParseTriplet(s, 0)
             else Result:=ParseTriplet(s, 1);
  end;

  function TAuMangaPage.ParseTriplet(s: string;  ersi: integer): string;
  var
    A: array of string;
    T: TStringList;
    r: string;
    j, L: integer;
    tex, prec, rad, atk: string;
    rtot: float;
  begin
    //    
    L:=0;
    SetLength(a, 1);
    rtot:=0;
    For j:=1 to Length(s) - 1 do begin
      case s[j] of
        '(': begin inc(L);
            if L > 1 then
              ComplainForError('  ' + ers1[ersi] +',   ','  ','');
          end;
        ')': begin
            dec(L);
            SetLength(a, Length(a) + 1);
          end;
        else
          if L = 1 then a[High(a)]:= a[High(a)] + s[j];
      end;
    end;
    if Length(a) = 0 then ComplainForError(' -    ' + ers2[ersi],'    ',',      ?..');
    T:=TStringList.Create;
    
    if ersi < 2 then begin
    // "", " ":
      For L:=0 to Length(a) - 1 do begin
        tex:=StrParm(a[L], 0, [',']);
        rad:=StrParm(a[L], 1, [',']);
        atk:=StrParm(a[L], 2, [',']);
        if rad = '' then if L = 0 then rad:='0' else rad:='1';
        if atk='' then atk:='0.7';
        if tex = '' then tex:='%0';
        tex:=ColorTextureCheck(tex);
        if tex[1] = '%' then tex:='%' + IntToStr(ColorToIntVerbose(tex, false));
        if not StrToFloatValid(rad, MinBorderDiameter, MaxBorderDiameter) then ComplainForGenericError(format(
          ' %d-     ' + ers2[ersi] + '    "%s"  .     %d.00  %d.00.',
          [L, rad, MinBorderDiameter, MaxBorderDiameter]));
        if (L > 0) and not StrToFloatValid(rad, 0.01, MaxBorderDiameter) then ComplainForGenericError(format(
          ' %d-     ' + ers2[ersi] + '       "'+rad+'",     0-  (..  ).',[L]));
        if not StrToFloatValid(atk, 0, 20) then ComplainForGenericError(format(
          ' %d-     ' + ers2[ersi] + '    "%s"  . '+
          '     0.00  20.00. (         ).', [L, atk]));
        T.Add(format('%s,%s,%s',[tex, rad, atk]));
        rtot:=rtot + StrToFloatVerbose( rad, MinBorderDiameter, MaxBorderDiameter);
      end;
      if rtot < 0 then ComplainForGenericError(
        '      ' + ers2[ersi] + '  ,  !');
    End
    else begin
    // "  ":
      For L:=0 to Length(a) - 1 do begin
        tex:=StrParm(a[L], 0, [',']);
        prec:=StrParm(a[L], 1, [',']);
        rad:=StrParm(a[L], 2, [',']);
        if rad = '' then rad := '2';
        if prec= '' then prec:= '40';
        tex:=ColorTextureCheck(tex);
        if (tex = '') or (tex[1] <> '%') then ComplainForGenericError(format(
          '  %s    "%s"  %d.   ,   ,      .'
           , [ers2[ersi], tex, L + 1]));
        tex:= IntToStr(ColorToIntVerbose(tex, false));
        if not StrToIntValid(rad, 0, 10) then ComplainForGenericError(format(
          '    %d-    %s    "%s".      0  10.'
           ,[L + 1, ers2[ersi], rad]));
        if not StrToIntValid(prec, 0, 255) then ComplainForGenericError(format(
          '   %d-    %s    "%s".     0  255.'
           ,[L + 1, ers2[ersi], prec]));
        T.Add(format('%s,%s,%s',[tex, prec, rad]));
      end;
    end;
    SetLength(a, 0);
    r:='';
    For j:=0 to T.Count-1 do r:=r + T[j] + ',';
    T.Free;
    Result:=r;
  end;

  function TAuMangaPage.ParseBalon(s: string): string;
  var
    j, u: integer;
    fn: string;
  begin
    Result:='';
    if s = '' then exit;
    s:=StrReplace(StrDelSet(s, [#10]), #13, '%');
    if s[Length(s)]= Tagmarker then ComplainForSyntaxError('  "%< >"!');
    For j:=1 to Length(s) do begin
      if s[j]= Tagmarker then begin
        inc(j);
        if not (UpCase(s[j]) in ValidTags) then
          ComplainForSyntaxError('  "' + Tagmarker + UpCase(s[j])+'"!');
        Result:=Result  + Tagmarker + UpCase(s[j]);
        if s[j]=TagImbeddedFile then begin
          fn:='';
          inc(j);
          if j > Length(s) then ComplainForError(
            '    "' + Tagmarker + TagImbeddedFile + '"', ' ','');
          if s[j]<>'(' then
            if s[j] = ' ' then ComplainForSyntaxError(
              ',     "' + Tagmarker + TagImbeddedFile + '"     .')
            else ComplainForError(
              '    "' + Tagmarker + TagImbeddedFile + '"', ' "' + s[j] + '"','');
          inc(j);
          While (j <= Length(s)) and (s[j] <> ')') do begin
            if j > Length(s) then ComplainForError(
              '       "' + Tagmarker + TagImbeddedFile + '"', ' ','');
            fn:=fn + s[j];
            if s[j] = '(' then ComplainForSyntaxError(
              '      "' + Tagmarker + TagImbeddedFile + '"  .'#10#13' .');
            inc(j);
          end;
          fn:=Trim(fn);
          For u:=0 to Parent.TextureName.High do
            if ParNameToCanonicForm(fn) = Parent.TextureName[u] then begin
              fn:=Parent.Texture[u];
              if (fn > '') and (fn[1] = '%') then ComplainForGenericError(
                '      "' + Tagmarker + TagImbeddedFile + '"   "' + Parent.TextureName[u] + '",    ("' + fn + '"),     . -.   >:(');
              Break;
            end;
          Result:=Result + '(' + Parent.Dir + fn + ')';
        end;
      end
      else Result:=Result + s[j];
    end;
  end;

  procedure  TAuMangaPage.Parse;
  var
    j: integer;
    Cmd: TCommand;
    Pars, FileOutName: string;
    AnyChanged, SmearChanged: boolean;
  begin
    Write('.');
    Try
      CurrentLine:=0;
      CurrentPos:=1;
      While CurrentLine < Count do begin
        Cmd:=StrToCommand(ExtractEntryName);
        ApplyCommand(Cmd, ExtractCommandPars);
      end;
    Except
      if ExceptObject is Exception
        then YellAndDie('    "%s":'#10#13'%s',
               [ExtractFileName(FileName), (ExceptObject as Exception).Message])
        else YellAndDie('     "%s",   -  , .',
               [ExtractFileName(FileName)]);
    End;
  end;

  procedure  TAuMangaPage.Execute;
  var
    j: integer;
    Cmd: TCommand;
    Pars, FileOutName: string;
    AnyChanged, SmearChanged: boolean;
    Scale: float;
  begin
    Try
      FileOutName:=Parent.OutputDir + ExtractFileName(FileName);
      AnyChanged:=(GetFileOutDate(FileOutName) <> Parent.Summaries.ReadString(FileName, 'Output file age', ''))
               or (Summary <> Parent.Summaries.ReadString(FileName, 'Summary', ''))
               or (Parent.Gmode > Parent.Summaries.ReadInteger(FileName, 'Quality', -1));
      SmearChanged:=(Parent.Gmode <> Draft) and (SmearSummary <> Parent.Summaries.ReadString(FileName, 'SmartSmear summary', ''));

      if AnyChanged or SmearChanged
      then begin
        WriteRu('  ...');
        if InColor then Scan:=TAuColorMangaScan.Create(FileName)
                   else Scan:=TAuMangaScan.Create(FileName);
        Write('...');

        if SmearChanged
        and not AnyChanged
        and not (Scan.hasAlpha or (StrDelSet(BalonsSmear.Text, [#10, #13]) <> ''))
          then WriteLnRu(' .')
          else AnyChanged:=True;
      end;

      if AnyChanged then begin
        Scan.SmearMethod:=SmearMethod;
        Scan.SmearPower:=SmearPower;

        if Cleaningpower > 0 then Scan.AutoClean(CleaningPower);
        if (PattToKill > 0) and (Parent.GMode <> Draft)
          then Scan.KillPatterns(PattToKill);
          
        Scale:=Scan.ScaleFactor(MinHeight, MinWidth, MaxHeight, MaxWidth);
        Case Parent.GMode of
          Draft: Scan.ApplySizeLimits(minf(Scale, 1));
          0: Scan.ApplySizeLimits(maxf(Scale, 1));
          Reliz: Scan.Upsample(Scale);
        end;

        if Scan.HasAlpha then Scan.SmearHoles;

        For j:=0 to SFX.Count - 1 do begin
          WriteRu('.');
          Scan.ProcessSFX(SFX[j]);
        end;
        
        For j:=0 to Balon.Count - 1 do begin
          WriteRu('.');
          Scan.MarkupBalon;
          Write('..');
          Scan.FitBalon(Balon[j], Parent.Font, BalonFontSize[j],  MinFontSize,
            FontCompression, Parent.Hardcore,  BalonBgr[j], BalonPrs[j],
            BalonSpacing[j]/100, HypenLen,   CompressionLoathing,
            ShortLinesLoathing,  TransitionsLoathing, DeviationsLoathing,
            BalonsSmear[j]);
        end;

        Case Parent.GMode of
          Draft: Scan.ApplySizeLimits(maxf(1, Scale));
          0: Scan.ApplySizeLimits(minf(1, Scale));
          Reliz: Scan.ApplySizeLimits(Scale / Scan.zoom);
        end;

        if LibPngDisabled then begin
          WriteRu('.  BMP');
          Scan.saveAsBmp(ChangeFileExt(FileOutName, '.bmp'))
        end
        else begin
          if not LibPngLoaded then LoadLibPNG;
          WriteRu('.');
          Scan.Save(FileOutName, PaletteDepth);
        end;
        Parent.Summaries.WriteString(FileName, 'Summary', Summary);
        Parent.Summaries.WriteString(FileName, 'Output file age', GetFileOutDate(FileOutName));
        Parent.Summaries.WriteInteger(FileName, 'Quality', Parent.Gmode);
        WriteLnRu('...OK.');
      end
      else WriteLnRu('   .');
      if Parent.GMode <> Draft then Parent.Summaries.WriteString(FileName, 'SmartSmear summary', SmearSummary)
    Except
      Parent.Summaries.UpdateFile;
      Parent.Summaries.Free;
      if ExceptObject is Exception
        then YellAndDie('    "%s":'#10#13'%s', [FileName, (ExceptObject as Exception).Message])
        else YellAndDie('     "%s",   -  , .',[FileName]);
    End;
  end;
  
  function GetCurrentDirWsep: string;
  begin
    Result:=GetCurrentDir;
    if (Result <> '') and (Result[Length(Result)] <> {$ifdef win32}'\'{$else}'/'{$endif})
    then Result:=Result + {$ifdef win32}'\'{$else}'/'{$endif};
  end;

  Constructor TAuManga.Create(FileName: string);
  var
    j: integer;
  begin
    inherited Create;
    if FileName = '' then YellAndDie(
      '       *.manga    !');
    _filename:=FileName;
    TextureName:=TAOS.Create;
    Texture:=TAOS.Create;
    Page:=TAOS.Create;
    PagePars:=TAOS.Create;
    Pages:=TAOO.Create;
    GlobalLevel:=True;
    CleaningPower:=0;
    paletteDepth:=4;
    FontSize:=17;
    MinFontSize:=8;
    FontCompression:=33;
    BalonColor:=255;
    BalonsPars:='%0,0,0.7';
    LineSpacing:=1.33;
    CompressionLoathing:=1.0;
    ShortLinesLoathing:=1.0;
    TransitionsLoathing:=1.0;
    DeviationsLoathing:=1.0;
    HypenLen:=7;
    SmearMethod:= STricky;
    if ParIs('-d') then begin
      SmearPower:= 0;
      GMode:=Draft;
    end
    else begin
      if ParIs('-r') then GMode:=Reliz;
      SmearPower:= 25;
    end;
    Dir:=ExtractFilePath(FileName);
    if  (Dir = '') or ({$ifdef win32} ExtractFileDrive(Dir) = ''{$else} Dir[1] <> '/'{$endif})
      then Dir:=GetCurrentDirWsep + Dir;
    if Dir[Length(Dir)] <> {$ifdef win32}'\'{$else}'/'{$endif} then Dir:=Dir + {$ifdef win32}'\'{$else}'/'{$endif};
    FileName:=Dir + ChangeFileExt(ExtractFileName(FileName), '.manga');
    if not FileExists(Filename) then YellAndDie(' "%s"  !',[FileName]);
    LoadFromFile(FileName);
    Summaries:= TMemIniFile.Create(ChangeFileExt(FileName, '.ini'));
    _BaseDir:=Dir;
    ClearSizeLimits;
    For j:=0 to Count-1 do Strings[j]:=Trim(Strings[j]);
    For j:=0 to Count-1 do Id[j]:= j + 1;
    //  ,  ,     
    j:=0;
    While (j < Count) do begin
      While (j < Count) and ((Strings[j] = '')  or (Strings[j][1] = ';'))
        do Delete(j);
      inc(j);
    end;
  end;

  Destructor TAuManga.Destroy;
  begin
    SafeFree(Texture);
    SafeFree(TextureName);
    SafeFree(Page);
    SafeFree(PagePars);
    SafeFree(Pages);
    if Assigned(Summaries) then Summaries.UpdateFile;
    SafeFree(Summaries);
  end;

  Procedure TAuManga.Parse;
  var
    Cmd: TCommand;
    Str, Str2: String;
    Page: TAuMangaPage;
    Prs: TStringListWithIDs;
    PageCount,  PBL, j: integer;
  begin
    WriteRu(#10#13#10#13' ' + UpperCase(ExtractFileName(_filename)) + ' ..');
    CurrentLine:=0;
    CurrentPos:=1;
      //  ...
      Str:=ExtractEntryName;
      if StrToCommand(Str) <> CBegin
        then ComplainForError('  "()"', Str, '');
      Prs:=ExtractCommandPars;
    OutputDir:=Prs.TextByOneLine;
    if OutputDir='' then OutputDir:=_BaseDir;
    {$ifdef win32}
    if ExtractFileDrive(OutputDir) = ''
    {$else}
    if OutputDir[1] <> '/'
    {$endif}
      then OutputDir:=_BaseDir + OutputDir;
//writeln(_basedir);
//writeln(outputdir);

    if OutputDir[Length(OutputDir)] = {$ifdef win32}'\'{$else}'/'{$endif}
      then OutputDir:=Copy(OutputDir, 1, Length(OutputDir) - 1);
    if not FileExists(OutputDir) then ComplainForGenericError(format(
'  "%s"  ,      .'#10#13+
'  , , ..',[OutputDir]));
    if OutputDir[Length(OutputDir)] <> {$ifdef win32}'\'{$else}'/'{$endif}
      then OutputDir:=OutputDir + {$ifdef win32}'\'{$else}'/'{$endif};
    if UpperCase(OutputDir) = UpperCase(_BaseDir) then  ComplainForGenericError(
' , ?!..       !..'#10#13+
'      !..');
    if (Pos('(', _BaseDir) > 0) or (Pos(')', _BaseDir) > 0) then ComplainForGenericError(
',    ,    ,    .'#10#13+
'  ("'+ _BaseDir + '"), , ...');
    PageCount:=0;
    While True do begin
      Str:=ExtractEntryName;
      if (Str <> '') and (Str[1]='%') then begin // 
        if not Assigned(Font[0]) then ComplainForGenericError(
          '   ,   "'+CommandsStr[CFont]+'()"    ');
        inc(PageCount);
        Str:=FileNamePrefix + Trim(StrDelSet(Str, ['%']));
        Prs:=ExtractCommandPars;
        Page:=TAuMangaPage.Create(Self, Prs, Str, Dir);
        Page.Parse;
        Pages.Add(Page);
      end
      else begin //
        Cmd:=StrToCommand(Str);
        Case Cmd of
          Cbegin: ComplainForGenericError(
'  "'+CommandsStr[Cmd]+'()"    ,   .');
          CBalon: ComplainForGenericError(
'    .'#10#13+
' (,       .)');
          CEnd: begin
            WriteLnRu('Ѩ ! :)'#10#13);
            Break;
          end;
        else
          Prs:=ExtractCommandPars;
          if (PageCount > 0) and (Cmd in CommandsNotAllowedBetweenPages)
           then ComplainForGenericError(
            ' "'+CommandsStr[Cmd]+'()"      .');
          Case Cmd of
            CHardcoreMania: Hardcore:=True;
            CDisablePng: LibPngDisabled:=True;
            else ApplyCommand(Cmd, Prs);
          end;
        end;
      end;
    End;

    WriteRu(#10#13' ...');
    For j:=0 to 9 do if Assigned(Font[j]) then Font[j].Load;
    WriteLnRu('OK.'#10#13);
    
    WriteLnRu(#10#13' ...'#10#13);
    For j:=0 to Pages.High do begin
      WriteLnRu (#10#13': ' + ExtractFileName((Pages[j] as TAuMangaPage).FileName));
      (Pages[j] as TAuMangaPage).Execute;
      Pages[j].Free;
    end;
  end;

  Constructor TParamsContainer.Create;
  begin
    Inherited;
    Balon:=TStringList.Create;
    BalonPrs:=TStringList.Create;
    BalonBGR:=TAOI.Create;
    SFX:=TStringList.Create;
    BalonFontSize:=TAOI.Create;
    BalonSpacing:=TAOI.Create;
    BalonsSmear:=TStringList.Create;
  end;

  Destructor TParamsContainer.Destroy;
  begin
    SFX.Free;
    Balon.Free;
    balonPrs.Free;
    balonBgr.Free;
    BalonFontSize.Free;
    BalonSpacing.Free;
    BalonsSmear.Free;
    Inherited;
  end;

   function TParamsContainer.StrToIntVerbose(str: string; Min, Max: integer): integer;
   begin
     str:=Trim(StrDelSet(str, ['%',#10, #13]));
     Try
       Result:=StrToInt(str);
       if (Result < Min) or (Result > Max) then Raise Exception.Create('');
     Except
       ComplainForError(format(
         '     %d  %d', [Min, Max]), Str, '');
     End;
   end;
   
   function TParserHelper.ParNameToCanonicForm(pm: string): string;
   var j: integer;
   begin
     pm:=UpperCase(pm);
     Result:='';
     For j:=1 to Length(pm) do begin
       case pm[j] of
         ' ', '_', #10, #13: if (result <> '') and (Result[Length(Result)] <> '_')
                Then Result:=Result + '_';
       else
         Result:=Result + pm[j];
       end;
     end;
     if (Result <> '') and (Result[Length(Result)] = '_') then Result:= Copy(Result, 1, Length(result));
   end;
   
   function TParamsContainer.StrToFloatVerbose(str: string; Min, Max: float): float;
//var j: integer;
   begin
//write(format('*** "%s"  %f  %f [',[str, min, max]));
//For j:=1 to Length(str) do write(ord(str[j]), ' ');
     str:=Trim(StrDelSet(str, ['%',#10, #13]));
     Try
       Result:=StrToFloat(str);
//writeln(format('] %f ',[result]));
       if (Result < (Min - 0.001)) or (Result > (Max + 0.001)) then Raise Exception.Create('');
     Except
//writeln((ExceptObject as Exception).Message);
       ComplainForError(format(
         '    %f  %f', [Min, Max]), Str, '');
     End;
   end;
   
   function TParamsContainer.StrToFloatValid(str: string; Min, Max: float): boolean;
   var f: float;
//j: integer;
   begin
//write(format('*** "%s"  %f  %f [',[str, min, max]));
//For j:=1 to Length(str) do write(ord(str[j]), ' ');
     str:=Trim(StrDelSet(str, ['%',#10, #13]));
     Try
       f:=StrToFloat(str);
//writeln(format('] %f ',[f]));
       if (f < (Min - 0.001)) or (f > (Max + 0.001)) then Result:=False
                                                     else Result:=True;
     Except
//writeln((ExceptObject as Exception).Message);
       Result:=false;
     End;
   end;
   
   function TParamsContainer.BitDpthToIntVerbose(str: string): integer;
   begin
     str:=Trim(StrDelSet(str, ['%',#10, #13]));
     Try
       Result:=StrToInt(str);
       if (Result <> 2) and (Result <> 4) and (result <> 8) then Raise Exception.Create('');
     Except
       ComplainForError('    2, 4  8', Str, '');
     End;
   end;
   
   function TParamsContainer.ColorToIntVerbose(str: string; TransparentAllowed: boolean): integer;
   var
     s1, s2, s3: string;
     m: integer;
   begin
     Try
       s1:=Trim(StrParm(str, 1, ['%']));
       s2:=Trim(StrParm(str, 2, ['%']));
       s3:=Trim(StrParm(str, 3, ['%']));
       if TransparentAllowed then m:=-1 else m:=0;
       if (s2 = '') and (s3 = '') then begin
         //volume
         Result:=StrToInt(s1);
         if (Result < m) or (Result > 255) then Raise Exception.Create('');
       end
       else Result:= $01000000
                   + StrToIntVerbose(s1, 0, 255) shl 16
                   + StrToIntVerbose(s2, 0, 255) shl 8
                   + StrToIntVerbose(s3, 0, 255);
     Except
       if TransparentAllowed then
         ComplainForError('     0  255,  -1,      0  255,  ''%''-', Str, '')
       else
         ComplainForError('     0  255,      0  255,  ''%''-', Str, '')
     End;
   end;
   

   function TParamsContainer.StrToIntValid(str: string; Min, Max: integer): boolean;
   begin
     Result:=True;
     Try
       StrToIntVerbose(str, Min, Max);
     Except
       Result:=False;
     End;
   end;
   
    Procedure TParamsContainer.ApplyCommand(Command: TCommand; Params: TStringListWithIDs);
    var
      AllowedSet: Set of TCommand;
      f, fi, fb, s, s2: string;
      i, j: integer;
    begin
      if GlobalLevel then AllowedSet:=CommandsAllowedGlobally
                     else AllowedSet:=CommandsAllowedInPage;
      if not (Command in AllowedSet) then begin
        if GlobalLevel then ComplainForGenericError(' "' + CommandsStr[Command]
          +'()"      !')
                       else ComplainForGenericError(' "' + CommandsStr[Command]
          +'()"     !');
      end;
      if not GlobalLevel and not (Command in CommandsAllowedBetweenBalons) then begin
        if Command in AlreadyApplied then ComplainForGenericError(
          ' "' + CommandsStr[Command]+'()"      !');
        if (Balon.Count > 0) or (SFX.Count > 0) then ComplainForGenericError(
          ' "' + CommandsStr[Command]+'()"       !');
        AlreadyApplied:=AlreadyApplied + [Command];
      end;
      Case Command of
        CTexture: With Self as TAuManga do begin
          s:=ParNameToCanonicForm(StrParm(Params.Text, 0, [',']));
          s2:=StrParm(Params.TextByOneLine, 1, [',']);
          if (s='') or (s2='') then
            ComplainForError('   "( <>, <  | %>)"',
                              format('" ( %s)"',[Params.TextByOneLine]), '');
          if pos('%', s) >0 then ComplainForSyntaxError('      .'#10#13'   .');
          if s2[1]='%' then ColorToIntVerbose(s2, false); //  
          j:=-1;
          For i:=0 to TextureName.High do
            if TextureName[i]=s then begin j:=i; Break end;
          if j < 0 then begin
            TextureName.Add(s);
            Texture.Add(s2);
          end
          else Texture[j]:=s2;
        end;
        CNamePrefix: FileNamePrefix:=Params.TextByOneLine;
        CFont: begin
            if Assigned((Self as TAuManga).Font[0]) then ComplainForGenericError(
              ' "()"   !     .');
            j:=0;
            While StrParm(Params.TextByOneLine, j, [',']) <>'' do begin
              if j > 9 then ComplainForGenericError('    !');
              (Self as TAuManga).Font[j]:=TFont.Create(_BaseDir + StrParm(Params.TextByOneLine, j, [',']));
              inc(j);
              Write('.');
            end;
            if j < 2 then ComplainForGenericError('  ()     :    ,   .');
          end;
        CFontSize: FontSize:=StrToIntVerbose(Params.TextByOneLine, MinimalFontSize, MaximalFontSize);
        CMinFontSize: MinFontSize:=StrToIntVerbose(Params.TextByOneLine, MinimalFontSize, MaximalFontSize);
        CCleaningPower: CleaningPower:=StrToIntVerbose(Params.TextByOneLine, 0, 100);
        CMaxFontCompression: FontCompression:=StrToIntVerbose(Params.TextByOneLine, 0, 70);
        CPaletteDepth: paletteDepth:=BitDpthToIntVerbose(Params.TextByOneLine);
        CMinWidth: begin
            MinWidth:=StrToIntVerbose(Params.TextByOneLine, MinPicDimensions, MaxPicDimensions);
            if MinWidth > MaxWidth then ComplainForGenericError(format(
             '    (%d)       (%d)!',[MinWidth,MaxWidth]));
          end;
        CMaxWidth: begin
            MaxWidth:=StrToIntVerbose(Params.TextByOneLine, MinPicDimensions, MaxPicDimensions);
            if MinWidth > MaxWidth then ComplainForGenericError(format(
             '    (%d)       (%d)!',[MaxWidth, MinWidth]));
          end;
        CMinHeight: begin
            MinHeight:=StrToIntVerbose(Params.TextByOneLine, MinPicDimensions, MaxPicDimensions);
            if MinHeight > MaxHeight then ComplainForGenericError(format(
             '    (%d)       (%d)!',[MinHeight,MaxHeight]));
          end;
        CMaxHeight: begin
            MaxHeight:=StrToIntVerbose(Params.TextByOneLine, MinPicDimensions, MaxPicDimensions);
            if MinHeight > MaxHeight then ComplainForGenericError(format(
             '    (%d)       (%d)!',[MaxHeight, MinHeight]));
          end;
        CDontScale: ClearSizeLimits;
        CScaleBefore: ScaleBeforeProcessing:=True;
        CScaleAfter: ScaleBeforeProcessing:=False;
        CColor: InColor:=True;
        CBlackAndWhite: InColor:=False;
        CSFX: begin
            if Balon.Count > 0 then ComplainForGenericError('      !');
            SFX.Add(ParseSFX(Params.TextByOneLine, True));
          end;
        CBalonsPars: BalonsPars:= ParseSFX(Params.TextByOneLine, True);
        CStdBalons: begin
            BalonColor:=255;
            BalonsPars:='%0,0,0.7';
          end;
        CStdNegaBalons: begin
            BalonColor:=0;
            BalonsPars:='%255,0,0.7';
          end;
        CBalonsBgr: BalonColor:=ColorToIntVerbose(ColorTextureCheck(Params.TextByOneLine), false);
        CTransparentBalonsBgr: BalonColor:=-1;
        CLineSpacing: LineSpacing:=StrToFloatVerbose(Params.TextByOneLine, 0.5, 5.0);
        CDisableAutoHypens: HypenLen:=MaxInt;
        CAutoHypens: HypenLen:=StrToIntVerbose(Params.TextByOneLine, 3, MaxInt);
        CCompressionLoathing: CompressionLoathing:=StrToFloatVerbose(Params.TextByOneLine, 0.0, 10.0);
        CShortLinesLoathing: ShortLinesLoathing:=StrToFloatVerbose(Params.TextByOneLine, 0.0, 10.0);
        CTransitionsLoathing: TransitionsLoathing:=StrToFloatVerbose(Params.TextByOneLine, 0.0, 10.0);
        CDeviationsLoathing: DeviationsLoathing:=StrToFloatVerbose(Params.TextByOneLine, 0.0, 10.0);
        CSmearMetod: SmearMethod:=StrToSmear(Params.TextByOneLine);
        CDraft: begin
            if (Self as TAuManga).GMode = Reliz then ComplainForGenericError(
              ' "()"  "()"      .          "-d"  "-r".');
            (Self as TAuManga).GMode:=Draft;
            SmearPower:=0;
          end;
        CReliz: begin
            if (Self as TAuManga).GMode = Draft then ComplainForGenericError(
              ' "()"  "()"      .          "-d"  "-r".');
            (Self as TAuManga).GMode:=Reliz;
          end;
        CSmearPower: if SmearPower > 0 then SmearPower:=StrToIntVerbose(Params.TextByOneLine, 1, 100);
        CSmearInBalons: BalonSmear:=ParseTriplet(Params.TextByOneLine, 2);
        CDontSmearInBalons: BalonSmear:='';
        CKillPatterns: PattToKill:=StrToIntVerbose(Params.TextByOneLine, 2, 5);
        CDontKillPatterns: PattToKill:=0;
        CBALON: begin
            Balon.Add(ParseBalon(Params.Text));
            BalonBgr.Add(BalonColor);
            BalonPrs.Add(BalonsPars);
            BalonFontSize.Add(FontSize);
            BalonSpacing.Add(Round(LineSpacing * 100));
            if (BalonColor < 0) and (InColor) //       
              then BalonsSmear.Add(BalonSmear)
              else BalonsSmear.Add('');
          end;
      end;
    end;

    function TParserHelper.ExtractEntryName: string;
    var s: string;
    begin
      MinErrLine:=-1;
      MaxErrLine:=-1;
//write('een');
//writeLn(format('*** %d  %d',[CurrentLine, CurrentPos]));
      If (CurrentLine < Count) and (CurrentPos > Length(Strings[CurrentLine]))
      then begin
        CurrentPos:=1;
        Inc(CurrentLine);
      end;
//writeln(format('&&& %d  %d  %d',[CurrentLine, CurrentPos, Count]));
      If CurrentLine >= Count then ComplainForError(
        '   ','  ...  , ',
        '      .'#10#13+
        ',      "()".');
      s:='';
      While (CurrentLine < Count)
      Do begin
//WriteLn(format('^^^ %d  %d  %s  "%s"',[CurrentLine, CurrentPos, Translit(Strings[CurrentLine][CurrentPos]), Translit(s)]));
//writeln('.',translit(strings[CurrentLine][CurrentPos]));
        if (s <> '') and (CurrentPos = 1) and (s[Length(s)] <> '_') then s:= s + '_'; //    
//writeln(Strings[CurrentLine][CurrentPos]);
        Case Strings[CurrentLine][CurrentPos] of
          ';': ComplainForSyntaxError('     !'#10#13'    ,      (";")'#10#13'.. ...      .');
          '(': break;
          ' ': if (s <> '') and (s[Length(s)] <> '_') then s:=s + '_';
          ')': begin
                 if GlobalLevel then s:='   ' else s:='';
                 ComplainForError(s, ' ',
                 '      .');
               end;
          else s:=s + Strings[CurrentLine][CurrentPos];
        end;
        Inc(CurrentPos);
        if CurrentPos > Length(Strings[CurrentLine]) then begin
          CurrentPos:=1;
          Inc(CurrentLine);
          if CurrentLine = Count then Break;
        end;
      end;
      if (s <> '') and (s[Length(s)] = '_') then Result:=Copy(s, 1, Length(s) - 1)
                                            else Result:=s;
//writelnru('"' + result + '"')      ;
    end;

    function TParserHelper.ExtractCommandPars: TStringListWithIDs;
    var
      s: string;
      bracketCount: integer;
    begin
//writeln('ecp');
      MinErrLine:=-1;
      MaxErrLine:=-1;
      Result:=TStringListWithIDs.Create;
      If (CurrentLine < Count) and (CurrentPos > Length(Strings[CurrentLine]))
      then begin
        CurrentPos:=1;
        Inc(CurrentLine);
      end;
      While (CurrentLine < Count) and (Strings[CurrentLine][CurrentPos] = ' ')
      do begin
        inc(CurrentPos);
        if CurrentPos > Length(Strings[CurrentLine])
        then begin
          CurrentPos:=1;
          Inc(CurrentLine);
        end;
      end;
      If CurrentLine = Count then ComplainForError(
        ' ','  ...  , ',
        '      .');
      if Strings[CurrentLine][CurrentPos] = ';'
        then ComplainForSyntaxError('     !'#10#13'    ,      (";")'#10#13'.. ...      .');
      if Strings[CurrentLine][CurrentPos] <> '('
        then ComplainForError(' ', '"' + Copy(Strings[CurrentLine], CurrentPos, Length(Strings[CurrentLine])) + '"','');
      inc(CurrentPos);
      if CurrentPos > Length(Strings[CurrentLine]) then begin
        CurrentPos:=1;
        Inc(CurrentLine);
      end;
      bracketCount:=1;
      s:='';
      While (CurrentLine < Count) and (bracketCount >= 1)
      do begin
        case Char(Strings[CurrentLine][CurrentPos]) of
          '(': inc (bracketCount);
          ')': begin
              dec (bracketCount);
                if bracketCount = 0 then begin
                if Trim(s) <> '' then begin
                  Result.Add(Trim(s));
                  Result.Id[Result.Count - 1]:=Id[CurrentLine];
                end;
                Inc(CurrentPos);
                Break;
              end;
            end;
        end;
        if bracketcount > 0 then s:=s + Strings[CurrentLine][CurrentPos];
        inc(CurrentPos);
        if CurrentPos > Length(Strings[CurrentLine]) then begin
          if (bracketcount >0) and (Trim(s) <> '') then begin
            Result.Add(Trim(s));
            Result.Id[Result.Count - 1]:=Id[CurrentLine];
            s:='';
          end;
          CurrentPos:=1;
          Inc(CurrentLine);
        end;
      end;
      if Result.Count > 0 then begin
        MinErrLine:=Result.Id[0];
        MaxErrLine:=Result.Id[Result.Count - 1];
      end;
      If CurrentLine >= Count then begin
        if GlobalLevel and (BracketCount > 0) then begin
          ComplainForError(format(
          'c      ,'#10#13'  %d  ',[bracketCount]),
          ' ,   - ,  ',
          '       .');
        end  
      end;
      if (CurrentLine < Count) and (CurrentPos > Length(Strings[CurrentLine])) then begin
        CurrentPos:=1;
        Inc(CurrentLine);
      end;
      //if Result.Count=0 then Result.Add(' ');
//writeLn(format('*** %d  %d',[CurrentLine, CurrentPos]));
//writelnru('"' + Result.Text + '"');
    end;


  Function TParserHelper.StrToCommand(s: string): TCommand;
  var j: TCommand;
  begin
    s:=UpperCase(s);
    For j:=Low(TCommand) to High(TCommand) do
      if CommandsStr[j]=s then begin
        Result:=j;
        Break;
      end;
      if j = NotACommand then ComplainForUnknownCommand(s);
    if Result = CAltBalon then Result:=CBalon;
  end;
  

  Function TParserHelper.StrToSmear(s: string): TSmear;
  var
    j: TSmear;
    ers: string;
  begin
    s:=UpperCase(Trim(s));
    For j:=Low(TSmear) to High(TSmear) do
      if SmearMethodName[j]=s then begin
        Result:=j;
        Exit;
      end;
    ers:='';
    For j:=Low(TSmear) to High(TSmear) do begin
      if SmearMethodName[j] = '' then Continue;
      ers:=ers + '"' + SmearMethodName[j] + '"';
      if ord(j) < ord(High(TSmear)) - 1 then ers:=ers + ', ';
      if ord(j) = ord(High(TSmear)) - 1 then ers:=ers + ' ';
    end;
    ComplainForSyntaxError(format(
      '   "%s".'{#10#13'   (%s)  : %s'}
      ,[s, VersionString, ers]));
  end;
  
  procedure TParamsContainer.ClearSizeLimits;
  begin
    MaxWidth:=MaxInt;
    MaxHeight:=MaxInt;
    MinWidth:=MinPicDimensions;
    MinHeight:=MinPicDimensions;
  end;

  function CyrillicUpCase(h: char): char;
  begin
    Case h of
      ''..'': Result:=chr(ord(h)+(ord('')-ord('')));
      '': Result:='';   '': Result:='';
      '': Result:='';   '': Result:='';
      '': Result:='';   '': Result:='';
      '': Result:='';   '': Result:='';
      '': Result:='';   '': Result:='';
      '': Result:='';   '': Result:='';
      '': Result:='';   '': Result:='';
      '': Result:='';
    else
      Result:=h;
    end;
  end;
  
  function UpCase(h: char): char;
  begin
    Result:=CyrillicUpCase(System.UpCase(h));
  end;


  Function UpperCase(s: string):string;
  var j: integer;
  begin
    For j:=1 to Length(s) do s[j]:=UpCase(s[j]);
    Result:=s;
  end;

  function TParserHelper.ErrorLocation: string;
  begin
    if MaxErrLine < 0
      then Result := format('  %d', [Id[CurrentLine]])
    else
      if MaxErrLine = MinErrLine
        then Result := format('  %d', [MaxErrLine])
        else Result := format('-   %d..%d', [MinErrLine, MaxErrLine]);
  end;
  
  procedure TParserHelper.ComplainForError(expected, whatwegot, suggestion: string);
  begin
    if suggestion = ''
      then Raise Exception.Create(Format(
       '  %s.'#10#13': %s,'#10#13' : %s',
       [ErrorLocation, expected, whatwegot]))
      else Raise Exception.Create(Format(
       '  %s.'#10#13': %s,'#10#13' : %s'
        +#10#13': %s',[ErrorLocation, expected, whatwegot, suggestion]));
  end;

  procedure TParserHelper.ComplainForUnknownCommand(c: string);
  begin
    Raise Exception.Create(Format(
      '  %s.'#10#13'  "%s".',
       [ErrorLocation, c]));
  end;

  procedure TParserHelper.ComplainForGenericError(s: string);
  begin
    Raise Exception.Create(Format(
      ' %s.'#10#13'%s', [ErrorLocation, s]));
  end;

  procedure TParserHelper.ComplainForSyntaxError(s: string);
  begin
    Raise Exception.Create(Format(
      '  %s.'#10#13'%s', [ErrorLocation, s]));
  end;


initialization
  DecimalSeparator:='.';
finalization

End.
