{
    This file is part of the Automaton program,
    Copyright (c) 2004 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the graphic processing classes.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

{$mode delphi}
{$macro on}
unit aut_graph;

interface

uses SysUtils, Classes, cl_typedefs, IniFiles, cl_classes, cl_strings, cl_dyna, cl_kambipng;


Const
  {$include aut_tags.inc}
  {$include aut_errorpic.inc}
  
  SFXWeightTreshold = 30;
  SfxWeightDiff = 15;
  MinimumMarkupBrightness = 30;
  NonBreakableSpaceWidth = 16;
  MinMarkupTreshold = 5;
  KoeffMultiplier = 100;
  MaxProbableSFXThickness = 100;
  RecursionLimit = 170000; //do NOT increase!..
  MinPicDimensions = 200;
  MaxPicDimensions = 4100;
  NBSP: char = #$A0;
  FontZoomTagMultiplier = 1.33;
  FitZoomMultiplier = 1.17;
  
Type
  TSmear = (SDraft, SSimple, SHard, SCircular, SSmarter, STRicky, SPicky, SSelective);

Const
  Yes = True;
  No = False;
  MinAlphaToSmear = 20;
  MaxSourceAlpha = 0.05;
  EdgeThoroughness = 1.9;
  DTRK = pi / 180;
  SmearAngleMultiplier:                array[TSmear] of float = (  1,    1,     1, 360/103,     1,  360/103,  360/103, 360/103);
  SmearStep:                           array[TSmear] of float = (  0, 1.41,     1,       5,   0.7,     0.45,      0.45,   0.45);
  MaximumSmearRadiusDiff:              array[TSmear] of float = (  0,  100,    15,     100,   100,       50,       50,      30);
  SmearDepth:                          array[TSmear] of float = (  0,    4,     0,      15,    10,       20,       10,      10);
  SmearConsiderEdges:                array[TSmear] of boolean = ( No,   No,    No,      No,   Yes,      Yes,      Yes,     Yes);
  SmearUseExponentialDistWeightFunc: array[TSmear] of boolean = ( No,   No,   Yes,      No,    No,       No,      Yes,     Yes);
  SmearNumberOfRays:                 array[TSmear] of integer = (  0,    0,     0,       0,     0,        7,        2,       1);
  SmearMinimumAngleBetweenRays:      array[TSmear] of integer = (  0,    0,     0,       0,     0,       30,      120,       0);
  SmearUseExponentialrayweight:      array[TSmear] of boolean = ( No,   No,    No,      No,    No,      Yes,       No,      No);
  SmearEdgeDetectDepth:              array[TSmear] of integer = (  0,    0,     0,       0,     1,        2,        2,       1);
  SmearEdgeDetRayMultiplier:           array[TSmear] of float = (  0,    0,     0,       0,     0,       10,       10,      10);

Type

  TAuGraph = class (T2DAOI)
  private
    _filename: string;
    MinBrightness, MaxBrightness: integer;
    procedure Transform; VIRTUAL;
    function _readwidth: integer;
    function _readheight: integer;
    procedure _ReadFromTGA(Filename: string);
    procedure _ReadFromPNG(Filename: string);
    procedure _FindLevels;
    procedure _CleanJPEGArtefacts (Power, Brmin, Brmax: integer);
    TransparentCount: integer;
    procedure _DetectEdges(Alpha, Edges: TAuGraph; di: integer);
    procedure MarkCorners; VIRTUAL;
    function __Upsample: TAuGraph;
  public
    Constructor Create; OVERLOAD;
    Constructor Create(fileName: string); OVERLOAD; VIRTUAL;
    Destructor Destroy; OVERRIDE;
    Procedure SaveAsBMP(FileName: string);  VIRTUAL;
    procedure Save(FileName: string; Colors: integer); VIRTUAL;
    Procedure _SaveAsBMP(FileName: string; Duh: T2DAOI);
    function CutChar(basey, basex, height, width: integer): TAuGraph;
    Procedure Scale(newHeight, NewWidth: integer); VIRTUAL;
    procedure Imbed (G: TAuGraph; y, x: integer); VIRTUAL;
    procedure ImbedDarkest (G: TAuGraph; y, x: integer); VIRTUAL;
    procedure TurnWhite;
    procedure Fill (v: integer);
    Property Height: integer read _readheight;
    Property Width: integer read _readwidth;
    Procedure Inverse;
    property FileName: string read _filename;
    function Inter(y, x: float): integer;
    function LInter(y, x: float): integer;
    function Pixel(y, x: integer): integer;// performs safe clamping
    function Blur(Alpha: TAuGraph; y, x, r: integer): integer; //returns blurred pixel
    function Deviation(A: TAuGraph; y, x, depth: integer): integer; //for edges detecting
    function HasAlpha: boolean;
    function Clone: TAuGraph; VIRTUAL;
  end;
  
  TAuTexture = class(TAuGraph)
  private
    function _Ixel(y, x: integer): integer; VIRTUAL;
  public
    function Texel(y, x: float): integer;
  end;

  TAuColorTexture = class(TAuTexture)
  private
    Procedure Transform; OVERRIDE;
    function _Ixel(y, x: integer): integer; OVERRIDE;
    function _IxelR(y, x: integer): integer;
    function _IxelG(y, x: integer): integer;
    function _IxelB(y, x: integer): integer;
  public
    function TexelR(y, x: float): integer;
    function TexelG(y, x: float): integer;
    function TexelB(y, x: float): integer;
  end;
  
  TFont = class; //forward declaration

  TArrayofFont = array[0..9] of TFont;
  
  TJustify = (JLeft, JRight, JCenter, JWidth);

  TWord = class(TAuGraph)
  public
    str: string;
    Interval: integer;
    Zoom, ZoomByFont: float;
    EndsParagraph, IsHypenSign: boolean;
    Justify, VertJustify: TJustify;
    Text: string;
  end;

  {$define header}
  {$define cl_objecttype}
    {$define typeofit := TWord}
    {$define typename := TWords}
      {$include cl_dyna_1template.inc}
  {$undef cl_objecttype}
  {$undef header}
  
  TPhrase = class
  private
    Words: TWords;
  public
    Justify, VertJustify: TJustify;
    EndsParagraph, EndsTheLine, WithOptionalHypen: boolean;
    Compression: integer;
    hpos, vpos, mincharsize: integer;
    zoom: float;
    Text: string;
    Constructor Create;
    Destructor Destroy; OVERRIDE;
    Procedure Add(w: TWord);
    function Height(zoom: float): integer;
    function Width(zoom: float; MinSize,  Compression: integer; EndsTheLine: boolean): integer;
    function IsLegal(zoom: float; MinSize: integer): boolean; VIRTUAL;
    Function Graph: TWord; VIRTUAL;
    Function vWidth(EndsTheLine: boolean): integer; VIRTUAL;
    Function vHeight: integer; VIRTUAL;
    Function vSpacing: integer; VIRTUAL;
  end;
  
  TVertPhrase = class(TPhrase)
    function Width(zoom: float): integer;
    function Height(zoom: float; MinSize, Compression: integer; EndsTheLine: boolean): integer;
    function IsLegal(zoom: float; MinSize: integer): boolean; OVERRIDE;
    Function Graph: TWord; OVERRIDE;
    Function vWidth(EndsTheLine: boolean): integer; OVERRIDE;
    Function vHeight: integer; OVERRIDE;
    Function vSpacing: integer; OVERRIDE;
  end;

  TFont = class
  private
    CharWidth, CharHeight: array [0..255] of integer;
    Intervals, IntervalsV: array[0..255, 0..255] of integer;
    IntervalsCalculated, IntervalsVCalculated: array[0..255, 0..255] of boolean;
    Chars: array[0..256] of TAuGraph;
    _name: string;
    _zoom: float;
    function Interval(a, b: char): integer;
    function IntervalV(a, b: char): integer;
  public
    procedure Load;
    Constructor Create (Name: string);
    Function Line(s: string): TWord;
    Function LineLen(s: string): integer;
    Function VertLine(s: string): TWord;
    Function VertLineLen(s: string): integer;
    Destructor Destroy; OVERRIDE;
    Property Name: string read _name;
    function Zoom: float;
  end;

  TLoathingArray = array [0..70] of array of Integer;

  {$define header}
  {$define cl_objecttype}
    {$define typeofit := TPhrase}
    {$define typename := TPhrases}
      {$include cl_dyna_1template.inc}
  {$undef cl_objecttype}
  {$undef header}
  
  TBalon = class(TAuGraph)
  private
    _BBeg, _BEnd: integer;
    Phrases: TPhrases;
    Bmin, Bmax: array of Integer;
    Loathingfactor: TLoathingArray;
    Procedure MarkBorders; VIRTUAL;
    procedure BeatStrToPhrases(str: string; F: TArrayOfFont;  Charsize: integer;
                                           Vertical: boolean; HypenLen: integer);
    function TryFitPhrases(Initialvpos: integer; zoom: float;
                   mincharsize, Compression: integer; spacing: float;
                     CompressionLoathing,  ShortLinesLoathing,
                     TransitionsLoathing, DeviationsLoathing: float): integer;
    function LeftBB(vpos, hgt: integer): integer;
    function WidthBB(vpos, hgt: integer): integer;
    Procedure ReallyPutPhrases; VIRTUAL;
  public
    BalonNum: integer;
    Constructor Create(G: TAuGraph; ymin, xmin, ymax, xmax: integer);
    Procedure Put(s: string; F: TArrayOfFont;   Charsize, mincharsize,
       Compression: integer; InitialZoom, spacing: float; HypenLen: integer;
         CompressionLoathing,  ShortLinesLoathing,  TransitionsLoathing, DeviationsLoathing: float);
    Destructor Destroy; OVERRIDE;
  end;

  TVertBalon = class(TBalon)
    Procedure ReallyPutPhrases; OVERRIDE;
  private
    Procedure MarkBorders; OVERRIDE;
  end;

  THardcoreBAlon = class(TVertBalon)
    Procedure ReallyPutPhrases; OVERRIDE;
  private
    Procedure MarkBorders; OVERRIDE;
  end;
  
  TCorner = record
    y, x: float;
  end;
  
  TCornerMarkup = array[0..2, 0..2] of TCorner;
  
  TAuMangaScan = class (TAuGraph)
  private
    Green, Red, Alpha, Koeff, Edges: TAuGraph;
    xmin,xmax,ymin,ymax, sfxbasevalue, sfx_diameter_marked,
    sfxcount, baloncount: integer;
    _InColor,
    _NowWorkingWithBalons, _edgesdetected: boolean;
    _corners: TCornerMarkup;
    _cornersmarked: boolean;
    procedure MarkCorners; OVERRIDE;
    procedure Transform; OVERRIDE;
    procedure SFXMarkupRecursiveFill (x, y: integer);
    procedure BalonMarkupRecursiveFill (x, y: integer);
    procedure ExpandSFXMarkup(diameter: integer);
    procedure ExpandSFXMarkupToNegativeSide;
    procedure _DrawSFX(G: TAuGraph; diameter, blur: float; brightness: integer);
    procedure FillBalonBackground(color: integer; SmearCmd: string); VIRTUAL;
    procedure _SmearHoles(B, G, R: TAuGraph; Method: TSmear; Power: integer);
    procedure LoadMarkupExternal;
    procedure DetectEdges(di: integer); VIRTUAL;
    procedure _Upsample; VIRTUAL;
    function _Deform(G: TAuGraph; NewHeight, NewWidth: integer): TAuGraph;
    Procedure _MarkupError(G: TAuGraph; treshold: integer); VIRTUAL;
  public
    zoom: float;
    EffectiveWidth, EffectiveHeight: integer;
    SmearMethod: TSmear;
    SmearPower, recursion: integer;
    destructor Destroy; OVERRIDE;
    Procedure MarkupSFX(diameter: integer);
    procedure DrawSFX(diameter, blur: float; brightness: integer); OVERLOAD; VIRTUAL;
    procedure DrawSFX(diameter, blur: float; texturefile: string); OVERLOAD; VIRTUAL;
    Procedure MarkupBalon;
    Procedure ReportError(G: TAuGraph;  treshold: integer; errormessage: string);
    procedure ProcessSFX(comline: string);
    Procedure Scale(newHeight, NewWidth: integer); OVERRIDE;
    Procedure ApplySizeLimits(sc: float);
    Procedure PutBalon(B: TBalon; Color: integer; Comline, SmearCmd: string);
    Procedure FitBalon(s: string; F: TArrayOfFont;
       Charsize, mincharsize, Compression: integer; Hardcore: boolean;
       Color: integer; Comline: string; spacing: float; HypenLen: integer;
       CompressionLoathing,  ShortLinesLoathing,  TransitionsLoathing,
       DeviationsLoathing: float; SmearCmd: string);
    procedure SmearHoles; VIRTUAL;
    procedure AutoClean (Power: integer); VIRTUAL;
    procedure KillPatterns(thickness: integer); VIRTUAL;
    function ScaleFactor(MinHeight, MinWidth, MaxHeight, MaxWidth: integer): float;
    procedure Upsample(sc: float);
    procedure Deform(NewHeight, NewWidth: integer); VIRTUAL;
  end;
  
  TAuColorMangaScan = class (TAuMangaScan)
  private
    RV, GV: TAuGraph;
    procedure Transform; OVERRIDE;
    procedure FillBalonBackground(color: integer; SmearCmd: string); OVERRIDE;
    procedure DetectEdges(di: integer); OVERRIDE;
    procedure MarkCorners; OVERRIDE;
    procedure _Upsample; OVERRIDE;
    Procedure _MarkupError(G: TAuGraph; treshold: integer); OVERRIDE;
  public
    procedure KillPatterns(thickness: integer); OVERRIDE;
    procedure Save(FileName: string; Colors: integer); OVERRIDE;
    Procedure SaveAsBMP(FileName: string); OVERRIDE;
    destructor Destroy; OVERRIDE;
    procedure DrawSFX(diameter, blur: float; brightness: integer); OVERRIDE;
    procedure DrawSFX(diameter, blur: float; texturefile: string); OVERRIDE;
    Procedure Scale(newHeight, NewWidth: integer); OVERRIDE;
    procedure AutoClean (Power: integer); OVERRIDE;
    procedure SmearHoles; OVERRIDE;
    function Clone: TAuGraph; OVERRIDE;
    procedure Deform(NewHeight, NewWidth: integer); OVERRIDE;
  end;


  Function YellAndDie(s: string); OVERLOAD;
  Function YellAndDie(s: string; c: array of const); OVERLOAD;
  
  function maxf (a,b:float):float;
  function minf (a,b:float):float;
  function min (a,b:integer):integer;
  function max (a,b:integer):integer;
  function clamp (x, min, max: integer):integer;

implementation


  function maxf (a,b:float):float;begin if a>b then Result:=a else Result:=b end;
  function minf (a,b:float):float;begin if a<b then Result:=a else Result:=b end;
  function max (a,b:integer):integer;begin if a>b then Result:=a else Result:=b end;
  function min (a,b:integer):integer;begin if a<b then Result:=a else Result:=b end;
  function clamp (x, min, max: integer):integer;begin if x<min then Result:=min else if x>max then Result:=max else Result:=x end;
  function ColorToRed(c: integer): integer; begin if c > $00FFFFFF then Result:=(c shr 16) and $FF else Result:=c end;
  function ColorToGreen(c: integer): integer; begin if c > $00FFFFFF then Result:=(c shr 8) and $FF  else Result:=c end;
  function ColorToBlue(c: integer): integer; begin  if c > $00FFFFFF then Result:=c and $FF else Result:=c end;
  function ColorToGray(c: integer): integer; begin
    if c > $00FFFFFF then Result:=Round(ColorToRed(c)*0.3 + ColorToGreen(c)*0.5 + ColorToBlue(c)*0.2)
                     else Result:=c
  end;
  
  function dist(y1, x1, y2, x2: float): float;
  begin
    dist:=Sqrt(sqr(y2 - y1) + sqr(x2 - x1));
  end;

  {$define cl_objecttype}
    {$define typeofit := TPhrase}
    {$define typename := TPhrases}
      {$include cl_dyna_1template.inc}
    {$define typeofit := TWord}
    {$define typename := TWords}
      {$include cl_dyna_1template.inc}
  {$undef cl_objecttype}

  function AngDiffAbs(a1, a2: float): float;
  begin
    Result:=a1 - a2;
    While Result < -180 do Result:=Result + 180;
    While Result > 180 do Result:=Result - 180;
    Result:=abs(Result);
  end;

{  function SmearAlphaWeightFactor(alphafactor: float): float;
  begin
    Result:= 1 - sqr(alphafactor * 2.1);
    if Result < 0 then Result:= 0;
  end;}
  function SmearDistWeightFactor(dist: float; Method: TSmear): float;
  begin
    if SmearUseExponentialDistWeightFunc[Method]
      then Result:= 1 / exp(dist)
      else Result:= 1 / (dist * dist * dist);
  end;
  
  function RayWeightFactor(ddist: float; Method: TSmear): float;
  begin
    if SmearUseExponentialrayweight[Method] then Result:= Exp(Exp(ddist/3))
                                           else Result:= ddist * ddist;// Exp(ddist);
  end;


  function TAuGraph.Pixel(y, x: integer): integer;
  begin
    Result:=D[ clamp( y, 0, Height - 1)][ clamp( x, 0, Width - 1)];
  end;
  
  function TAuGraph.Deviation(A: TAuGraph; y, x, depth: integer): integer;
  var
    dev, dx, dy, ddx, ddy: integer;
  begin
    Result:=0;
    if A.Pixel(y, x) > MinAlphaToSmear then Exit;
    dev:=0;
    For dx:=-depth to depth do
      For dy:=-depth to depth do
        if (dx <> 0) and (dy <> 0)
        and (A.Pixel(y + dy, x + dx) < MinAlphaToSmear)
          then begin
            inc(Result, round(abs((Pixel(y + dy, x + dx) - Pixel(y, x))) / (sqr(dx) + sqr(dy))));
            inc(dev);
          end;
      if dev = 0 then result:=0
    else Result:=Result * (sqr(depth * 2 + 1)) div dev;
  end;
  
  function TAuGraph.Inter(y, x: float): integer;
  var
    fy, fx, val, weight: float;
    iy, ix: integer;
  begin
    iy:=Trunc(y);
    ix:=Trunc(x);
    fy:=Frac(y);
    fx:=Frac(x);
    val:= sqr((1 - fy)*(1 - fx)) * Pixel(iy,     ix)
        + sqr((1 - fy)*   fx   ) * Pixel(iy,     ix + 1)
        + sqr(   fy   *   fx   ) * Pixel(iy + 1, ix + 1)
        + sqr(   fy   *(1 - fx)) * Pixel(iy + 1, ix);
    weight:=sqr((1 - fy)*(1 - fx))
          + sqr((1 - fy)*   fx   )
          + sqr(   fy   *   fx   )
          + sqr(   fy   *(1 - fx));
    Result:= round( val / weight);
  end;
  
  function TAuGraph.LInter(y, x: float): integer;
  var
    fy, fx, val, weight: float;
    iy, ix: integer;
  begin
    iy:=Trunc(y);
    ix:=Trunc(x);
    fy:=Frac(y);
    fx:=Frac(x);
    val:= ((1 - fy)*(1 - fx)) * Pixel(iy,     ix)
        + ((1 - fy)*   fx   ) * Pixel(iy,     ix + 1)
        + (   fy   *   fx   ) * Pixel(iy + 1, ix + 1)
        + (   fy   *(1 - fx)) * Pixel(iy + 1, ix);
    weight:=((1 - fy)*(1 - fx))
          + ((1 - fy)*   fx   )
          + (   fy   *   fx   )
          + (   fy   *(1 - fx));
    Result:= round( val / weight);
  end;
  
  procedure TAuGraph._DetectEdges(Alpha, Edges: TAuGraph; di: integer);
  var  r, x, y, dx, dy, d0, px, py, tot, q: integer;
       v: float;
  begin
    q:=sqr(di + 1);
    For y:=0 to Height - 1 do begin
      if (y div 100) <> ((y - 1) div 100) then Write('.');
      For x:=0 to Width - 1 do begin
        r:=0; //tot:=0;
        For dy:=-di to di do
          For dx:=-di to di do
            if (x + dx >= 0) and (y + dy >= 0) and (x + dx < width) and (y + dy < height)
            and (Alpha[y + dy][x + dx] < MinAlphaToSmear)
            and (sqr(dx) + sqr(dy) <= q)
            then r:= max(r, abs(D[y + dy][x + dx] - D[y][x]));
        Edges[y][x]:=r * 4;
      end;
    end;
  end;

  function TAuGraph.Blur(Alpha: TAuGraph; y, x, r: integer): integer; //returns blurred pixel
  var e, dx, dy, tot, q: integer;
  begin
    e:=0;
    tot:=0;
    For dy:=-r to r do
      For dx:=-r to r do
        if (x + dx >= 0) and (y + dy >= 0) and (x + dx < width) and (y + dy < height)
        and (Alpha[y + dy][x + dx] < MinAlphaToSmear)
        and (sqr(dx) + sqr(dy) <= sqr(r))
        then begin
          inc(e, D[y + dy][x + dx]);
          inc(tot);
        end;
     Result:=abs (e div max(1, tot));
  end;

  procedure TAuMangaScan.DetectEdges(di: integer);
  begin
    if _edgesdetected then Exit;
    writeRu('. ');
    _DetectEdges(Alpha, Edges, di);
    _edgesdetected:=True;
  end;
  
  procedure TAuColorMangaScan.DetectEdges(di: integer);
  var  r, x, y, dx, dy, d0, px, py, tot, q: integer;
       v: float;
  begin
    if _edgesdetected then Exit;
    writeRu('. ');
    q:=sqr(di + 1);
    For y:=0 to Height - 1 do begin
      if (y div 100) <> ((y - 1) div 100) then Write('.');
      For x:=0 to Width - 1 do begin
        r:=0; //tot:=0;
        For dy:=-di to di do
          For dx:=-di to di do
            if (x + dx >= 0) and (y + dy >= 0) and (x + dx < width) and (y + dy < height)
            and (Alpha[y + dy][x + dx] < MinAlphaToSmear)
            and (sqr(dx) + sqr(dy) <= q)
            then begin
              r:= max(r, abs(D[y + dy][x + dx] - D[y][x]));
              r:= max(r, abs(RV[y + dy][x + dx] - RV[y][x]));
              r:= max(r, abs(GV[y + dy][x + dx] - GV[y][x]));
            end;
        Edges[y][x]:=r * 4;
      end;
    end;
    _edgesdetected:=True;
  end;
  
  procedure TAuMangaScan.KillPatterns(thickness: integer);
  var
    M, A, E: TAuGraph;
    x, y: integer;
    ky, f: float;
  begin
    WriteRu('..( ..');
    M:= Self.Clone;
    M.Scale(Height div thickness, Width div thickness);
    A:= Alpha.Clone;
    A.Scale(Height div thickness, Width div thickness);
    E:= TAuGraph.Create;
    E.SetRect(Height div thickness, Width div thickness);
    M._DetectEdges(A, E, 1);
    For y:=0 to Height - 1 do begin
      ky:= y / height * M.Height;
      if (y div 100) <> ((y - 1) div 100) then Write('.');
      For x:=0 to width - 1 do begin
        f:=clamp(E.Inter(ky, x / Width * E.Width), 0, 255) / 255;
        D[y][x]:=round(f * D[y][x] + (1 - f) * Self.Blur(Alpha, y, x, thickness));
      end;
    end;
    E.Free;
    A.Free;
    M.Free;
    WriteRu('.).');
  end;

  procedure TAuColorMangaScan.KillPatterns(thickness: integer);
  var
    M, A, E: TAuGraph;
    x, y: integer;
    ky, f: float;
  begin
    WriteRu('..( ..');
    M:= Self.Clone;
    M.Scale(Height div thickness, Width div thickness);
    A:= Alpha.Clone;
    A.Scale(Height div thickness, Width div thickness);
    E:= TAuGraph.Create;
    E.SetRect(Height div thickness, Width div thickness);
    M._DetectEdges(A, E, 1);
    For y:=0 to Height - 1 do begin
      ky:= y / height * M.Height;
      if (y div 100) <> ((y - 1) div 100) then Write('.');
      For x:=0 to width - 1 do begin
        f:=clamp(E.Inter(ky, x / Width * M.Width), 0, 255) / 255;
        D[y][x]:=round(f * D[y][x] + (1 - f) * Self.Blur(Alpha, y, x, thickness));
        RV[y][x]:=round(f * RV[y][x] + (1 - f) * RV.Blur(Alpha, y, x, thickness));
        GV[y][x]:=round(f * GV[y][x] + (1 - f) * GV.Blur(Alpha, y, x, thickness));
      end;
    end;
    E.Free;
    A.Free;
    M.Free;
    WriteRu('.).');
  end;


  {$rangechecks off}
  procedure TAuMangaScan._SmearHoles(B, G, R: TAuGraph; Method: TSmear; Power: integer);
  var
    x, y, progressCount, ToDo, nl, totalrays: integer;
    anglestep, weight, value1, value2, value3, NoiseLevel: float;
    rval1, rval2, rval3, rwgt, rang: array[0..725] of float;
    procedure ShowProgress;
    begin
      inc(ProgressCount);
      if ProgressCount > Width then begin
        write('.');
        {if (((ToDo - ProgressCount) * 10) div TransparentCount)
        <> ((ToDo * 10) div TransparentCount)
          then write((ToDo * 10) div TransparentCount);}
        Dec(ToDo, progressCount);
        ProgressCount:=0;
      end;
    end;

    // TRICKY SMEAR vvvvv *************************
    procedure SmearItTricky;
    var angle, alphafactor, edgefactor, currentweight, border,
        begvalue1, begvalue2, begvalue3, rayvalue1, rayvalue2, rayvalue3,
        val1, val2, val3, rayweight, wm, flatweight,
        fx, fy, dist, InitialEdgeDensity, minradius: float;
        i, j, rtot: integer;
    begin
      totalrays:= 1 + round( pi*2 / AngleStep);
      Assert(totalrays < 725, 'bug: total rays >= 725');
      angle:=0;
      weight:=0;
      value1:=0;
      value2:=0;
      value3:=0;
      minradius:=max(width, height);
      //     __ ( ):
      While angle < pi*2 do begin
        dist:= SmearStep[Method];
        fx:= x;
        fy:= y;
        rayvalue1:=0;
        rayvalue2:=0;
        rayvalue3:=0;
        rayweight:=0;
        border:=max(Width, Height);
        begvalue1:=0;
        begvalue2:=0;
        begvalue3:=0;
        val1:=0;
        val2:=0;
        val3:=0;
        While (fx > 0) and (fy > 0)
        and (fx < Width) and (fy < Height)
        and (dist < border + smearDepth[Method] / 3)
        and (dist - minradius < MaximumSmearRadiusDiff[Method])
        do begin
          fx:=x + dist * sin(angle);
          fy:=y + dist * cos(angle);
          alphafactor:=Alpha.Inter(fy, fx) / 255;
          if alphafactor < MaxSourceAlpha then begin
            edgefactor:=Edges.Inter(fy, fx);
            if (edgefactor > (NoiseLevel * 4))
            or ((dist < border)
                and ((sqr(val1 - begvalue1)*0.4 + sqr(val2 - begvalue2) + sqr(val3 - begvalue3)*0.6) > sqr(NoiseLevel)*4))
            then break;
            currentweight:= SmearDistWeightFactor(dist, SPicky);{SmearAlphaWeightfactor(alphafactor) *}
            rayweight:=rayweight + currentweight;
            val1:= B.Inter(fy, fx);
            if Assigned(G) then begin
              val2:= G.Inter(fy, fx);
              val3:= R.Inter(fy, fx);
            end;
            if (dist < border) then begin
              begvalue1:=val1;
              begvalue2:=val2;
              begvalue3:=val3;
              border:=dist;
            end;
            rayvalue1:=rayvalue1 + currentweight * val1;
            if Assigned(G) then begin
              rayvalue2:=rayvalue2 + currentweight * val2;
              rayvalue3:=rayvalue3 + currentweight * val3;
            end;
          end;
          dist:=dist + SmearStep[Method];
        end;
        if border < minradius then minradius:=border;
        if border > dist then border:=dist;
        value1:=value1 + rayvalue1;// * RayWeightFactor(dist - border, SPicky);
        value2:=value2 + rayvalue2;// * RayWeightFactor(dist - border, SPicky);
        value3:=value3 + rayvalue3;// * RayWeightFactor(dist - border, SPicky);
        weight:=weight + rayweight;// * RayWeightFactor(dist - border, SPicky);
        angle:= angle + 3 * AngleStep;
      end;
//write('*');
{      //   ,   
      flatweight:=weight; if flatweight = 0 then flatweight:=1;
      angle:=0;
      i:=0;
      While angle < pi*2 do begin
        dist:= SmearStep[Method];
        fx:= x;
        fy:= y;
        rayvalue1:=0;
        rayvalue2:=0;
        rayvalue3:=0;
        rayweight:=0;
        border:=max(Width, Height);
        begvalue1:=0;
        begvalue2:=0;
        begvalue3:=0;
        val1:=0;
        val2:=0;
        val3:=0;
        While (fx > 0) and (fy > 0)
        and (fx < Width) and (fy < Height)
        and (dist < border + smearDepth[Method] / 3)
        and (dist - minradius < MaximumSmearRadiusDiff[Method])
        do begin
          fx:=x + dist * sin(angle);
          fy:=y + dist * cos(angle);
          alphafactor:=Alpha.Inter(fy, fx) / 255;
          if alphafactor < MaxSourceAlpha then begin
            edgefactor:=Edges.Inter(fy, fx);
            if (edgefactor < (NoiseLevel * 2))
            or ((dist > border) and ((edgefactor < InitialEdgeDensity / EdgeThoroughness)
                                or (edgefactor > InitialEdgeDensity * EdgeThoroughness)))
            //or ((sqr(val1 - begvalue1)*0.4 + sqr(val2 - begvalue2) + sqr(val3 - begvalue3)*0.6) > sqr(NoiseLevel)* 4)
            then
            break;
            currentweight:= SmearAlphaWeightfactor(alphafactor) * SmearDistWeightFactor(dist, Method);
            rayweight:=rayweight + currentweight;
            val1:= B.Inter(fy, fx);
            if Assigned(G) then begin
              val2:= G.Inter(fy, fx);
              val3:= R.Inter(fy, fx);
            end;
            if (dist < border) then begin
              begvalue1:=val1;
              begvalue2:=val2;
              begvalue3:=val3;
              border:=dist;
              InitialEdgeDensity:=edgefactor;
            end;
            rayvalue1:=rayvalue1 + currentweight * val1;
            if Assigned(G) then begin
              rayvalue2:=rayvalue2 + currentweight * val2;
              rayvalue3:=rayvalue3 + currentweight * val3;
            end;
          end;
          dist:=dist + SmearStep[Method];
        end;
        if border < minradius then minradius:=border;
        if border > dist then border:=dist;
        rval1[i]:=rayvalue1 * RayWeightFactor(dist - border, Method);
        rval2[i]:=rayvalue2 * RayWeightFactor(dist - border, Method);
        rval3[i]:=rayvalue3 * RayWeightFactor(dist - border, Method);
        rwgt[i]:=rayweight * RayWeightFactor(dist - border, Method);
        rang[i]:=angle * 180 / Pi;
        inc(i);
        angle:= angle + AngleStep;
      end;
//write('=');
      //    ,      ...
      rtot:=0;
      Repeat
        j:=-1;
        wm:=0;
        For i:=0 to totalrays - 1 do begin
          if wm < rwgt[i] then begin
            wm:=rwgt[i];
            j:=i;
          end;
        end;
        if j >=0 then begin
          For i:=0 to totalrays - 1 do
            if (i <> j) and (AngDiffAbs(rang[j], rang[j]) < SmearMinimumAngleBetweenRays[Method])
              then rwgt[i]:=0;
          currentweight:=flatweight * SmearEdgeDetRayMultiplier[Method] / SmearNumberOfRays[Method];
          weight := weight + rwgt[j] * currentweight;
          value1:= value1 + rval1[j] * currentweight;
          value2:= value2 + rval2[j] * currentweight;
          value3:= value3 + rval3[j] * currentweight;
          inc(rtot);
        end;
      Until (j < 0) or (rtot = SmearNumberOfRays[Method]);}
//write('-');
    end;
    // TRICKY SMEAR ^^^^^^^^^^ *********************



    // SMARTER SMEAR vvvvv *************************
    procedure SmearItSmarter;
    var angle, alphafactor, edgefactor, currentweight, border,
        rayvalue1, rayvalue2, rayvalue3, rayweight,
        fx, fy, dist, InitialEdgeDensity, minradius: float;
    begin
      angle:=0;
      weight:=0;
      value1:=0;
      value2:=0;
      value3:=0;
      minradius:=max(width, height);
      While angle < pi*2 do begin
        dist:= SmearStep[Method];
        fx:= x;
        fy:= y;
        rayvalue1:=0;
        rayvalue2:=0;
        rayvalue3:=0;
        rayweight:=0;
        edgefactor:=0;
        InitialEdgeDensity:=0;
        border:=max(Width, Height);
        While (fx > 0) and (fy > 0)
        and (fx < Width) and (fy < Height)
        and (dist < border + smearDepth[Method])
        and (dist - minradius < MaximumSmearRadiusDiff[Method])
        do begin
          fx:=x + dist * sin(angle);
          fy:=y + dist * cos(angle);
          alphafactor:=Alpha.Inter(fy, fx) / 255;
          if alphafactor < MaxSourceAlpha then begin
            if SmearConsiderEdges[Method] then edgefactor:=Edges.Inter(fy, fx);
            // ,       ,
            // ,     :
            if (dist < border) then begin
              border:=dist;
              //    -     
              if SmearConsiderEdges[Method] then InitialEdgeDensity:= edgefactor;
            end;
            if SmearConsiderEdges[Method] then begin
              //,     ,
              //     __  ,
              //     :
              if (edgefactor < InitialEdgeDensity / EdgeThoroughness)
              or (edgefactor > InitialEdgeDensity * EdgeThoroughness)
              then Break;
            end;
            currentweight:= SmearDistWeightFactor(dist, Method); //SmearAlphaWeightfactor(alphafactor) * // * (edgefactor * SmearEdgesWeight + SmearFlatsWeight);
            rayweight:=rayweight + currentweight;
            rayvalue1:=rayvalue1 + currentweight * B.Inter(fy, fx);
            if Assigned(G) then begin
              rayvalue2:=rayvalue2 + currentweight * G.Inter(fy, fx);
              rayvalue3:=rayvalue3 + currentweight * R.Inter(fy, fx);
            end;
          end;
          dist:=dist + SmearStep[Method];
        end;
        if border < minradius then minradius:=border;
        if border > dist then border:=dist;
        value1:=value1 + rayvalue1 * RayWeightFactor(dist - border, Method);
        value2:=value2 + rayvalue2 * RayWeightFactor(dist - border, Method);
        value3:=value3 + rayvalue3 * RayWeightFactor(dist - border, Method);
        weight:=weight + rayweight * RayWeightFactor(dist - border, Method);
        angle:= angle + AngleStep;
      end;
    end;
    // SMARTER SMEAR ^^^^^^^^^^ *********************

    // SIMPLE SMEAR vvvvvvvvvv  *********************
    procedure SmearItSimple;
    var angle, alphafactor, currentweight, border,
        fx, fy, dist, minradius: float;
    begin
      angle:=0;
      weight:=0;
      value1:=0;
      value2:=0;
      value3:=0;
      minradius:=max(width, height);
      While angle < pi*2 do begin
        dist:= SmearStep[Method];
        fx:= x;
        fy:= y;
        border:=max(Width, Height);
        While (fx > 0) and (fy > 0)
        and (fx < Width) and (fy < Height)
        and (dist < border + smearDepth[Method])
        and (dist - minradius < MaximumSmearRadiusDiff[Method])
        do begin
          fx:=x + dist * sin(angle);
          fy:=y + dist * cos(angle);
          alphafactor:=Alpha.Inter(fy, fx) / 255;
          if alphafactor < MaxSourceAlpha then begin
            // ,       ,
            // ,     :
            if (dist < border) then border:=dist;
            currentweight:= SmearDistWeightFactor(dist, Method);//SmearAlphaWeightfactor(alphafactor)  * (edgefactor * SmearEdgesWeight + SmearFlatsWeight);
            weight:=weight + currentweight;
            value1:=value1 + currentweight * B.Inter(fy, fx);
            if Assigned(G) then begin
              value2:=value2 + currentweight * G.Inter(fy, fx);
              value3:=value3 + currentweight * R.Inter(fy, fx);
            end;
          end;
          dist:=dist + SmearStep[Method];
        end;
        if border < minradius then minradius:=border;
        if border > dist then border:=dist;
        angle:= angle + AngleStep;
      end;
    end;
      // SIMPLE SMEAR ^^^^^ ****************

      // DRAFT SMEAR vvvvv ****************
    procedure SmearItDraft;
    var
      alphafactor, currentweight: float;
      j, dir: integer;
    begin
      weight:=0;
      value1:=0;
      value2:=0;
      value3:=0;
      For dir := -1 to 1 do
        if dir <> 0 then begin
          j:=x;
          While (j > 0) and (j < Width - 1) do begin
            inc(j, dir);
            if Alpha[y][j] < MaxSourceAlpha * 255 then begin
              currentweight:=SmearDistWeightFactor(abs(j - x), Method);
              value1:=value1 + B[y][j] * currentweight;
              weight:=weight + currentweight;
              if Assigned(G) then begin
                value2:=value2 + currentweight * G[y][j];
                value3:=value3 + currentweight * R[y][j];
              end;
            end;
          end;
        end;
    end;
      // DRAFT SMEAR ^^^^^^^^^^ ***************
  begin
    if TransparentCount = 0 then Exit;
    AngleStep:= 6.2832 / (round((Power + 3) * SmearAngleMultiplier[Method]));
    if SmearConsiderEdges[Method] then begin
      DetectEdges(SmearEdgeDetectDepth[Method]);
      nl:=0;
      For y:=0 to Height - 1 do
        For x:=0 to Width - 1 do
          inc(nl, clamp(edges[y][x], 0, 128));
      NoiseLevel:=nl / (Width * Height);
//edges.saveasbmp('__edges.bmp');
    end;
    Case Method of
      SDraft: writeRu('. ..');
      SSimple, SHard, SCircular: writeRu('...');
      SSmarter, STRicky, SPicky, SSelective: writeRu('. ..');
    end;
    ToDo:=TransparentCount;
    ProgressCount:=0;
    For y:=0 to Height - 1 do
      For x:=0 to Width - 1 do
        if Alpha[y][x] > MinAlphaToSmear then begin
          Case Method of
            SDraft: SmearItDraft;
            SSimple, SHard, SCircular: SmearItSimple;
            SSmarter: SmearItSmarter;
            STRicky, SPicky, SSelective: SmearItTricky;
          end;
          Try
            B[y][x]:=round(value1 / weight);
            if Assigned(G) then begin
              G[y][x]:=round(value2 / weight);
              R[y][x]:=round(value3 / weight);
            end;
          Except
            B[y][x]:=255 * ErrorPic[y and $7, x and $1F];
            if Assigned(G) then begin
              G[y][x]:=0;
               R[y][x]:=255 - B[y][x];
            end;
          End;
          if Method<>SDraft then ShowProgress;
        end;
  end;
  {$rangechecks off}

  Procedure TAuMangaScan.ReportError(G: TAuGraph; treshold: integer; errormessage: string);
  begin
    _MarkupError(G, treshold);
    WriteRu('. "_error.bmp"...');
    SaveAsBmp(ExtractFilePath(_filename) + '_error.bmp');
    WriteLnRu('OK.');
    YellAndDie(errormessage + #10#13#10#13 +
'         _error.bmp'#10#13+
' (  ,    )');
  end;
  

  Procedure TAuMangaScan._MarkupError(G: TAuGraph; treshold: integer);
  var y,x: integer;
  begin
    WriteRu('.  ..');
    For y:=0 to Height - 1 do begin
      if (y div 100) <> ((y - 1) div 100) then Write('.');
      For x:=0 to Width -1 do
        if G[y][x] > treshold
          then D[y][x]:=255 - 255 * ErrorPic[y and $7, x and $1F]
          else D[y][x]:=128 + (D[y][x] - 128) div 2;
    end;
  end;
  
  Procedure TAuColorMangaScan._MarkupError(G: TAuGraph; treshold: integer);
  var y,x: integer;
  begin
    WriteRu('.  ..');
    For y:=0 to Height - 1 do begin
      if (y div 100) <> ((y - 1) div 100) then Write('.');
      For x:=0 to Width -1 do
        if G[y][x] > treshold then begin
          D[y][x]:=255 * ErrorPic[y and $7, x and $1F];
          GV[y][x]:=0;
          RV[y][x]:=255 - D[y][x];
        end
        else begin
          D[y][x]:=128 + D[y][x] div 2;
          RV[y][x]:=128 + RV[y][x] div 2;
          GV[y][x]:=128 + GV[y][x] div 2;
        end;
    end;
  end;
  
  Const
    cYName: array[0..1] of string = (' ', ' ');
    cXName: array[0..2] of string = ('-', '-', '-');

  procedure TAuMangaScan.MarkCorners;
  var
    j, i, my, sx, mx, cy, cx: integer;
    eYell: string;
    found: array[0..1, 0..2] of boolean;
    function IsBlue(py, px: integer): boolean;
    begin
      Result:= (((D[py][px] shr 16)and $FF) - max(((D[py][px] shr 8)and $FF), (D[py][px] and $FF))) > 100;
    end;
  begin
    FillChar(found, sizeof(found), 0);
    For cy:=0 to 1 do begin
      if cy=0 then my:=1 else my:=-1;
      For cx:=0 to 2 do begin
        if cx = 0 then mx:=1 else mx:=-1;
        if cx = 0 then sx:=0 else sx:=1;
        j:=min (Height div 5, width div 5);
        repeat
          For i:=0 to j do begin
            Case cx of
              0, 2: begin
                  if IsBlue(cy * (Height - 1) + (j - i) * my, sx * (Width - 1) + i * mx) then begin
                    found[cy, cx]:=True;
                    _corners[cy, cx].x:=sx * (Width - 1) + i * mx;
                    _corners[cy, cx].y:=cy * (Height - 1) + (j - i) * my;
                    j:=0;
                    Break;
                  end;
                end;
              1: begin
                  if IsBlue(cy * (Height - 1) + (j - i) * my, (Width div 2) + i) then begin
                    found[cy, cx]:=True;
                    _corners[cy, cx].x:=(Width div 2) + i;
                    _corners[cy, cx].y:=cy * (Height - 1) + (j - i) * my;
                    j:=0;
                    Break;
                  end;
                  if IsBlue(cy * (Height - 1) + (j - i) * my, (Width div 2) - i) then begin
                    found[cy, cx]:=True;
                    _corners[cy, cx].x:=(Width div 2) - i;
                    _corners[cy, cx].y:=cy * (Height - 1) + (j - i) * my;
                    j:=0;
                    Break;
                  end;
                end;
            end;
          end;
          dec(j);
        until j < 0;
      end;
    end;
    For cy:=0 to 1 do For cx:=0 to 2 do _cornersmarked:=_cornersmarked or found[cy, cx];
    if not _cornersmarked then Exit;
//writeln;for cy:=0 to 1 do for cx:=0 to 2 do writeln(format('%d  %d  %f  %f',[cy, cx, _corners[cy, cx].y, _corners[cy, cx].x]));readln;
    eYell:='';
{    if (found[0, 1] and (not found[0, 0] or not found[0, 2]))
    or (found[1, 1] and (not found[1, 0] or not found[1, 2]))
    then eYell:=eYell + '         !'#10#13;}
    if (not found[0, 0] and not found[1, 2])
    or (not found[1, 0] and not found[0, 2])
    or (not found[1, 0] and not found[0, 0])
    or (not found[1, 2] and not found[0, 2])
    then eYell:=eYell + '        !'#10#13;
    if eYell <> '' then begin
      eYell:=eYell + '  : ';
      For cy:=0 to 1 do For cx:=0 to 2 do
        if found[cy, cx] then eYell:=eYell + cXName[cx] + cYName[cy];
      YellAndDie(eYell);
    end;
    //filling the undefined corners:
    if not found[0, 0] then begin
      _corners[0, 0].x:=_corners[1, 0].x + (_corners[0, 2].x - _corners[1, 2].x);
      _corners[0, 0].y:=_corners[1, 0].y + (_corners[0, 2].y - _corners[1, 2].y);
    end;
    if not found[1, 0] then begin
      _corners[1, 0].x:=_corners[0, 0].x + (_corners[1, 2].x - _corners[0, 2].x);
      _corners[1, 0].y:=_corners[0, 0].y + (_corners[1, 2].y - _corners[0, 2].y);
    end;
    if not found[0, 2] then begin
      _corners[0, 2].x:=_corners[1, 2].x + (_corners[0, 0].x - _corners[1, 0].x);
      _corners[0, 2].y:=_corners[1, 2].y + (_corners[0, 0].y - _corners[1, 0].y);
    end;
    if not found[1, 2] then begin
      _corners[1, 2].x:=_corners[0, 2].x + (_corners[1, 0].x - _corners[0, 0].x);
      _corners[1, 2].y:=_corners[0, 2].y + (_corners[1, 0].y - _corners[0, 0].y);
    end;
    if not found[0, 1] then begin
      _corners[0, 1].x:=(_corners[0, 0].x + _corners[0, 2].x ) / 2;
      _corners[0, 1].y:=(_corners[0, 0].y + _corners[0, 2].y ) / 2;
    end;
    if not found[1, 1] then begin
      _corners[1, 1].x:=(_corners[1, 0].x + _corners[1, 2].x ) / 2;
      _corners[1, 1].y:=(_corners[1, 0].y + _corners[1, 2].y ) / 2;
    end;
    EffectiveWidth:=Round((dist(_corners[0, 0].y, _corners[0, 0].x, _corners[0, 2].y, _corners[0, 2].x)
                         + dist(_corners[1, 0].y, _corners[1, 0].x, _corners[1, 2].y, _corners[1, 2].x)) / 2);
    EffectiveHeight:=Round((dist(_corners[0, 0].y, _corners[0, 0].x, _corners[1, 0].y, _corners[1, 0].x)
                         + dist(_corners[0, 2].y, _corners[0, 2].x, _corners[1, 2].y, _corners[1, 2].x)) / 2);
//writeln;for cy:=0 to 1 do for cx:=0 to 2 do writeln(format('%d  %d  %f  %f',[cy, cx, _corners[cy, cx].y, _corners[cy, cx].x]));readln;
  end;

  procedure TAuMangaScan.LoadMarkupExternal;
  var
    M: TAuMangaScan;
    x, y: integer;
  begin
    if //not (Self is TAuColorMangaScan) //     -  ! :)
    //and
    not FileExists(changefileext(changefileext(_filename, '') + '_', '.tga'))
    and not FileExists(changefileext(changefileext(_filename, '') + '_', '.png'))
    then Exit;
    writeRu('(  .');
    M:=TAuMangaScan.Create(_filename + '_');
    if (M.Width <> Width) or (M.Height <> Height) then YellAndDie(
'    "%s" (%dx%d)'#10#13+
'      ,     '#10#13+
' ("%s", %dx%d)' ,[M.FileName, M.Width, M.Height, _filename, Width, Height]);
    write('..');
    if Self is TAuColorMangaScan
    then For y:=0 to height - 1 do
           For x:=0 to width - 1 do begin
             Green[y][x]:=M.Green[y][x];
             Red[y][x]:=M.Red[y][x];
           end
    else For y:=0 to height - 1 do
           For x:=0 to width - 1 do begin
             Green[y][x]:=max(Green[y][x], M.Green[y][x]);
             Red[y][x]:=max(Red[y][x], M.Red[y][x]);
           end;
    if not Self._cornersmarked and M._cornersmarked then begin
      _corners:=M._corners;
      _cornersmarked:=True;
    end;
    M.Free;
    write('.)..');
  end;



  function TBalon.TryFitPhrases(Initialvpos: integer; zoom: float;
                   mincharsize, Compression: integer; spacing: float;
                     CompressionLoathing,  ShortLinesLoathing,
                     TransitionsLoathing, DeviationsLoathing: float): integer;
  var
    j, LoathingFactor, vpos, vheight, comp, pnum, CurrentLineLen, CurrentLineWidth: integer;
    Phrase: TPhrase;
    Compressionfactor: float;
    Function Fits: boolean;
    var wd: integer;
    begin
       wd:=CurrentLineWidth + Phrase.vWidth(True);
        if CurrentLineLen > 0 then inc(wd, round((Phrase.vHeight / 2)*Compressionfactor));
      Result:= wd < WidthBB(vpos, max(vheight, Phrase.vHeight));
    end;
    Procedure Add;
    begin
      vheight:=max(vheight, round(Phrase.vSpacing * spacing));
      inc(CurrentLineWidth, Phrase.vWidth(False));
      if CurrentLineLen > 0 then inc(CurrentLineWidth, round((Phrase.vHeight / 2)*Compressionfactor));
      inc(CurrentLineLen);
      inc(Loathingfactor, round(100 * CompressionLoathing * Compression));
    end;
    Procedure Next;
    //here we really       (- , ?..) ...
    var i, wlim, left, right, leftover, pos: integer;
    begin
      wlim:=WidthBB(vpos, vheight);
      left:=LeftBB(vpos, vheight);
      right:=left + wlim;
      Phrases[j].EndsTheLine:=True;
      if Phrases[j].WithOptionalHypen then Inc(LoathingFactor, round(50000 * TransitionsLoathing));
      if Self is THardcoreBalon then Phrases[j].Justify:=JLeft;
      case Phrases[j].Justify of
        JLeft: begin
            pos:=left;
            For i:=j - CurrentLineLen + 1 to j do begin
              Phrases[i].hpos:=pos;
              inc(pos, Phrases[i].vWidth(i = j));
              if not Phrases[i].WithOptionalHypen
                then inc(pos, round((Phrases[i].vHeight / 2)*(100 - Phrases[i].Compression)/100));
            end;
            leftover:=max(0, right - pos);
          end;
        JRight: begin
            pos:=right - Phrases[j].vWidth(True);
            For i:=j downto j - CurrentLineLen + 1 do begin
              Phrases[i].hpos:=pos;
              dec(pos, Phrases[i].vWidth(i = j));
              if (i > 0) and not Phrases[i - 1].WithOptionalHypen
                then dec(pos, round((Phrases[i].vHeight / 2)*(100 - Phrases[i].Compression)/100));
            end;
            leftover:=max(0, pos - left);
          end;
        JCenter, JWidth: begin
            leftover:=max(0, wlim - CurrentLineWidth);
            pos:=left + leftover div 2;
            For i:=j - CurrentLineLen + 1 to j do begin
              Phrases[i].hpos:=pos;
              inc(pos, Phrases[i].vWidth(i = j));
              if not Phrases[i].WithOptionalHypen
                then inc(pos, round((Phrases[i].vHeight / 2)*(100 - Phrases[i].Compression)/100));
            end;
          end;
{        JWidth: begin
            For i:=j - CurrentLineLen + 1 to j do begin
              ,  ...
            end;
          end;  }
      end;
      inc(LoathingFactor, round(500*ShortLinesLoathing * leftover/wlim));
      //  :
      For i:=j - CurrentLineLen + 1 to j do
        case Phrases[i].VertJustify of
          JLeft: Phrases[i].vpos:= vpos + (vheight - Phrases[i].vheight);
          JRight: Phrases[i].vpos:=vpos;
        else
          Phrases[i].vpos:=
                      vpos + (vheight - Phrases[i].vheight) div 2;
        end;
      inc(vpos, vheight);
      CurrentLineLen:=0;
      CurrentLineWidth:=0;
    end;
  begin
    Result:=maxInt;
    if Self is THardcoreBalon then spacing:=spacing + 0.5;
    Loathingfactor:=0;
    Compressionfactor:=(100 - Compression) / 100;
    vpos:=Initialvpos;
    vheight:=0;
    CurrentLineLen:=0;
    CurrentLineWidth:=0;
    For j:=0 to Phrases.High do with Phrases[j] do begin hpos:=0; vpos:=0; EndsTheLine:=false end;
    j:=0;
    While j <= Phrases.High do begin
      Phrase:=Phrases[j];
      Phrase.Compression:=Compression;
      Phrase.mincharsize:=mincharsize;
      Phrase.zoom:=zoom;
      if not Phrase.IsLegal(zoom, mincharsize) then YellAndDie (
'      %d,     %d%s,'#10#13+
'      (%d)'#10#13+
'   %d ("%s").', [BalonNum, Round(zoom * 100), '%', mincharsize, j + 1, Phrase.Text]);
      //Try to add to current line
      if Fits then Add
      else
        if CurrentLineLen = 0 then Exit //return MaxInt, i.e. "does not fit".
        else begin
          dec(j); Next; inc(j);
          if Fits then Add else Exit;
        end;
      if ((j = Phrases.High) or Phrase.EndsParagraph) then Next;
      inc(j);
    end;
    Inc(Loathingfactor, round(60000 *Abs((vpos + InitialVpos) - (_bend + _bbeg)) * DeviationsLoathing / (_bend - _bbeg)));
    Result:=Loathingfactor;
  end;

  function TBalon.LeftBB(vpos, hgt: integer): integer;
  var j, Amin: integer;
  begin
    Result:=-1;
    if (vpos < _bbeg) or ((vpos + hgt - 1) > _bend) then Exit;
    Amin:=Bmin[vpos];
    For j:=vpos + 1 to vpos + hgt - 1 do Amin:=max(Amin, Bmin[j]);
    Result:=Amin;
  end;

  function TBalon.WidthBB(vpos, hgt: integer): integer;
  var j, Amax: integer;
  begin
    Result:=-1;
    if (vpos < _bbeg) or ((vpos + hgt - 1) > _bend) or (LeftBB(vpos, hgt) < 0) then Exit;
    Amax:=Bmax[vpos];
    For j:=vpos + 1 to vpos + hgt - 1 do Amax:=min(Amax, Bmax[j]);
    Result:=Amax - LeftBB(vpos, hgt) - 1;
  end;

  Procedure TBalon.ReallyPutPhrases;
  var j: integer;
      G: TWord;
  begin
    For j:=0 to Phrases.High do begin
      With Phrases[j] do begin
        G:=Graph;
        Self.ImbedDarkest(G, vpos, hpos);
        G.Free;
      end;
    end;
  end;
  
  Procedure TVertBalon.ReallyPutPhrases;
  var j: integer;
      G: TWord;
  begin
    For j:=0 to Phrases.High do begin
      With Phrases[j] do begin
        G:=Graph;
        Self.ImbedDarkest(G, hpos, vpos);
        G.Free;
      end;
    end;
  end;
  
  Procedure ThardcoreBalon.ReallyPutPhrases;
  var j: integer;
      G: TWord;
  begin
    For j:=0 to Phrases.High do begin
      With Phrases[j] do begin
        G:=Graph;
        Self.ImbedDarkest(G, hpos, Self.Width - 1 - vpos - G.Width);
        G.Free;
      end;
    end;
  end;

  Procedure TBalon.Put(s: string; F: TArrayOfFont; Charsize, mincharsize,
        Compression: integer; InitialZoom, spacing: float; HypenLen: integer;
          CompressionLoathing,  ShortLinesLoathing,
          TransitionsLoathing, DeviationsLoathing: float);
  var
    j, c, u, y, optV, optC: integer;
    zoom: float;
  begin
    if s = '' then Exit;
    BeatStrToPhrases(s, F, Charsize, Self is TVertBalon, HypenLen);
    write('.');
    if Phrases.Length = 0 then Exit;
    zoom:= InitialZoom;
    While True do begin
      For j:=_Bbeg to _Bend do begin
        //          
        For c:=0 to Compression do
          LoathingFactor[c][j]:=TryFitPhrases(j, zoom, mincharsize, c, spacing,
             CompressionLoathing,  ShortLinesLoathing,  TransitionsLoathing, DeviationsLoathing);
        For c:=Compression + 1 to 70 do
          LoathingFactor[c][j]:=MaxInt;
      end;
      // ,    
      u:= MaxInt;
      For j:=_Bbeg to _Bend do  begin
        For c:=70 downto 0 do begin
          if Loathingfactor[c][j] < u then begin
            optC:=c;
            optV:=j;
            u:=Loathingfactor[c][j];
          end;
        end;
      end;
      if u < MaxInt then begin
        TryFitPhrases(optV, zoom, mincharsize, optC, spacing,
          CompressionLoathing,  ShortLinesLoathing,  TransitionsLoathing, DeviationsLoathing);
          //     ,  
          //     ...
        ReallyPutPhrases;
        Break;
      end;
      zoom:=zoom / FitZoomMultiplier;
      write('.');
      //  ,    TryFitPhrases
      //     ,   ,
      //  .
    end;
    //Phrases.FreeContained;
  end;

  Constructor TPhrase.Create;
  begin
    inherited;
    Words:=TWords.Create;
  end;

  Destructor TPhrase.Destroy;
  var
    j: integer;
  begin
    For j:=0 to Words.High do Words[j].Free;
    SafeFree(Words);
    inherited;
  end;

  Procedure TPhrase.Add(w: TWord);
  begin
    EndsParagraph:=w.EndsParagraph;
    Justify:=w.Justify;
    VertJustify:=w.Justify;
    Words.Add(w);
    Text:=Text + w.Text;
  end;

  function TPhrase.Height(zoom: float): integer;
  var  j: integer;
  begin
    Result:=0;
    For j:=0 to words.high do
      Result:=max ( Result, round(zoom * words[j].zoom * words[j].zoomByFont * words[j].Height));
  end;
  
  Function TPhrase.vSpacing: integer;
  var  j: integer;
  begin
    Result:=0;
    For j:=0 to words.high do
      Result:=max ( Result, round(zoom * words[j].zoom * words[j].Height));
  end;

  function TVertPhrase.Width(zoom: float): integer;
  var  j: integer;
  begin
    Result:=0;
    For j:=0 to words.high do
      Result:=max ( Result, round(zoom * words[j].zoom * words[j].zoomByFont * words[j].Width));
   end;

  function TVertPhrase.vSpacing: integer;
  var  j: integer;
  begin
    Result:=0;
    For j:=0 to words.high do
      Result:=max ( Result, round(zoom * words[j].zoom * words[j].Width));
   end;

  function TPhrase.Width(zoom: float; MinSize, Compression: integer; EndsTheLine: boolean): integer;
  var  j: integer;
  begin
    Result:=0;
    For j:=0 to words.high do
      if (not ( (j = words.high) and (words[j].IsHypenSign))) or EndsTheLine
        then inc(Result, max (
          round(zoom * words[j].zoom * words[j].zoomByFont
            * words[j].Width *((100 - Compression) / 100)),
          round((MinSize / words[j].Height) * words[j].width * words[j].zoomByFont) ) );
  end;

  function TVertPhrase.Height(zoom: float; MinSize, Compression: integer; EndsTheLine: boolean): integer;
  var  j: integer;
  begin
    Result:=0;
    For j:=0 to words.high do
      if (not ( (j = words.high) and (words[j].IsHypenSign))) or EndsTheLine
        then inc(Result, max (
          Round(zoom * words[j].zoom * words[j].zoomByFont
            * words[j].Height *((100 - Compression) / 100)),
          Round((MinSize / words[j].Width) * words[j].Height * words[j].zoomByFont) ) );
  end;
  
  Function TPhrase.vWidth(EndsTheLine: boolean): integer;
  begin
    Result:=Width(zoom, mincharsize, compression, EndsTheLine);
  end;
  
  Function TPhrase.vHeight: integer;
  begin
    Result:=Height(zoom);
  end;

  Function TVertPhrase.vWidth(EndsTheLine: boolean): integer;
  begin
    Result:=Height(zoom, mincharsize, compression, EndsTheLine);
  end;

  Function TVertPhrase.vHeight: integer;
  begin
    Result:=Width(zoom);
  end;

  function TPhrase.IsLegal(zoom: float; MinSize: integer): boolean;
  var j: integer;
  begin
    Result:=true;
    For j:=0 to Words.High do begin
//writeln(format('*** %d  %f  %f',[j, words[j].zoom, words[j].zoom * zoom * Words[j].Height]));
      if round(words[j].zoom * zoom * Words[j].Height) < MinSize
      then begin
        Result:=false;
        Break
      end;
    end;
  end;

  function TVertPhrase.IsLegal(zoom: float; MinSize: integer): boolean;
  var j: integer;
  begin
    Result:=true;
    For j:=0 to Words.High do
      if round(words[j].zoom * zoom * Words[j].Width) < MinSize
      then begin
        Result:=false;
        Break
      end;
  end;

  Function TPhrase.Graph: TWord;
  var
    j, pos, vpos: integer;
  begin
    Result:=TWord.Create;
    Result.SetRect(Height(zoom), Width(zoom, MincharSize, Compression, EndsTheLine));
    Result.TurnWhite;
    pos:=0;
    For j:=0 to words.high do begin
      Words[j].Scale(
        round(zoom * words[j].zoom * words[j].ZoomByFont * words[j].Height),
        max (round(zoom * words[j].zoom * words[j].ZoomByFont * words[j].Width *((100 - Compression) / 100))
            ,round((MincharSize / words[j].Height) * words[j].ZoomByFont * words[j].width ) ) );
      Case words[j].VertJustify of
        JLeft: vpos:=Result.Height - words[j].Height;
        JRight: vpos:=0;
      else
        vpos:=(Result.Height - words[j].Height) div 2;
      end;
      if (not (words[j].IsHypenSign)) or EndsTheLine
        then Result.ImbedDarkest(words[j], vpos, pos);
      inc(pos, words[j].Width);
      Words[j].Free;
    end;
//result.SaveAsBmp('()'+IntToStr(random(1000)) + '.bmp');
    //Self.Free;
  end;

  Function TVertPhrase.Graph: TWord;
  var
    j, pos, vpos: integer;
  begin
    Result:=TWord.Create;

    Result.SetRect(max(1,Height(zoom, MincharSize, Compression, EndsTheLine)), max(1,Width(zoom)));
    //w windowse rabotalo, a w linukse daet 0, i gotowo - access violation...

    Result.TurnWhite;
    pos:=0;
    For j:=0 to Words.High do begin
      Words[j].Scale( max(
        Round(zoom * words[j].zoom * words[j].ZoomByFont * words[j].Height *((100 - Compression) / 100)),
        Round((MincharSize / words[j].Width) * words[j].Height ) )
        , round(zoom * words[j].zoom * words[j].ZoomByFont * words[j].Width));
      Case words[j].VertJustify of
        JLeft: vpos:=Result.Width - words[j].Width;
        JRight: vpos:=0;
      else
        vpos:=(Result.Width - words[j].Width) div 2;
      end;
      if (not (words[j].IsHypenSign)) or EndsTheLine
        then Result.ImbedDarkest(words[j], pos, vpos);
//r.SaveAsBmp('()'+IntToStr(j) + '.bmp');
      inc(pos, words[j].Height);
      Words[j].Free;
    end;
//result.SaveAsBmp('()'+IntToStr(random(1000)) + '.bmp');
    //Self.Free;
  end;

  const
    RussianChars: set of Char = [''..'',''..'','',''];
    Vowels: set of Char = ['','','','','','','','','','',
                           '','','','','','','','','',''];

  function AddHypens(str: string): string;
  var
    j: integer;
    Consonants: set of Char;
  begin
    Consonants:= RussianChars - Vowels - ['','','','','',''];
    result:='';
    str:='  ' + str + '  ';
    For j:=3 to length(str) - 2 do begin
      if ( j > 5)
      and (str[j] in Consonants)
      and ((str[j-1] in Vowels)
           or ((str[j-1] in Consonants)
               and ((str[j-2] in Vowels) and not((str[j-3] = TagMarker)))))
      and (str[j+1] in Vowels)

      then Result:=Result + TagMArker + TagOptionalHypen;
      Result:=Result + str[j];
    end;
  end;

  function InserAutoHypens(str: string; HypenLen: integer): string;
  var
    s: string;
    j: integer;
  begin
    Result:='';
    For j:=1 to Length(str) do begin
      if (str[j] in RussianChars) or (str[j] = '-') then s:=s + str[j]
      else begin
        if length(s) >= HypenLen then Result:=Result + AddHypens(s)
                                 else Result:=Result + s;
        s:='';
        Result:=Result + str[j];
      end;
    end;
    if length(s) >= HypenLen then Result:=Result + AddHypens(s)
                             else Result:=Result + s;
  end;

  procedure TBalon.BeatStrToPhrases(str: string; F: TArrayOfFont;
                     Charsize: integer; Vertical: boolean; HypenLen: integer);
  var
    j, Fnum, NextInterval: integer;
    Justify, VertJustify: TJustify;
    z: float;
    s: string;
    Words: TWords;
    procedure Normalize;
    begin
      Justify:=JCenter;
      VertJustify:=JCenter;
      z:=Charsize / 32;
      Fnum:=0;
    end;
    procedure Next(EndsParagraph: boolean);
    begin
      if (s = '') and ((Words.Length = 0) or not EndsParagraph) then Exit;//     .
      if s = '' then s:=nbsp;
      if vertical then Words.Add(F[Fnum].VertLine(s))
                  else Words.Add(F[Fnum].Line(s));
//WriteRu(inttostr(words.High)+'  "'+s+'" ');
///riteLn(Words.Last.Endsparagraph);
//words.last.saveasbmp('__yt_'+IntToStr(words.high)+'.bmp');
      words.Last.EndsParagraph:=EndsParagraph;
      words.Last.Justify:=Justify;
      words.Last.VertJustify:=VertJustify;
      words.Last.Zoom:= z;
      words.Last.Interval:= NextInterval;
      words.Last.Text:=StrReplace(s, NBSP, '&nbsp;');
      NextInterval:=0;
//writeln ('"',translit(s),'" ',Fnum);
      s:='';
      if EndsParagraph then Normalize;
    end;
    procedure ExpandPhrases;
    begin
      if vertical then Phrases.Add(TVertPhrase.Create)
                  else Phrases.Add(TPhrase.Create);
    end;
  begin
    Words:=TWords.Create;
    Normalize;
    str:=StrReplace(str, ' - ', TagMarker + TagNonBreakableSpace + '- ');
    if not Vertical then str:=InserAutoHypens(str, HypenLen);
    s:='';
    For j:=1 to System.Length(str) do begin
      if str[j] = TagMarker then begin // ...
        inc(j);
        Case str[j] of
          TagImbeddedFile: begin // % -   (     )
              Next(False);
              inc(j, 2); s:='';
              While str[j] <> ')' do begin
                s:=s + str[j];
                inc(j)
              end;
              Words.Add(TWord.Create(s));
              words.Last.EndsParagraph:=False;
              words.Last.Justify:=Justify;
              words.Last.VertJustify:=VertJustify;
              words.Last.Zoom:= z;
              words.Last.ZoomByFont:=1.0;
              words.Last.Interval:= NextInterval;
              words.Last.Text:=s;
              NextInterval:=0;
              s:='';
            end;
          TagOptionalHypen: begin
              Next(False);
              s:='-';
              Next(False);
              words.Last.Interval:=0;
              words.Last.IsHypenSign:=True;
              words.Last.Text:='';
              words.Last.Justify:=Justify;
              words.Last.VertJustify:=VertJustify;
              NextInterval:=-1;
            end;
          TagVertical: ; //   . % -   ,       ,    
          TagHardcore: ; //   . % -   ,    ,  ,    
          TagEndParagraph: Next(True); // % -  
          '0'..'9': begin //%0..%9 -  
               Next(False);
               FNum:=StrToInt(str[j]);
             end;
          TagBig: z:= z * FontZoomTagMultiplier; // % -      
          TagSmall: z:= z / FontZoomTagMultiplier; // % -      
          TagNormalize: begin // % -       0- 
              Next(False);
              Normalize;
            end;
          TagBold: begin //  % -   (1-) 
              Next(False);
              FNum:=1;
            end;
          TagItalic: begin // % -   (2-) 
              Next(false);
              FNum:=2;
            end;
          TagJustifyLeft: Justify:=JLeft; // % -      
          TagJustifyRight: Justify:=JRight; // % - ..  
          TagJustifyCenter: Justify:=JCenter; // % - ..  
          TagJustifyWidth: Justify:=JWidth; // %' - ..  
          TagJustifyVertTop: VertJustify:=JRight; // % -        
          TagJustifyVertBottom: VertJustify:=JLeft; // % - .. 
          TagNonBreakableSpace: s:=s + NBSP; // % -  
          TagLeftBracket: s:= s + '('; // %[ -   
          TagRightBracket: s:= s + ')'; // %] -   
          TagPercent: s:=s + '%'; // %/ -   ("%").
        end;
      end
      else
        if str[j] in [' ', NBSP] then begin
          Next(false);
          NextInterval:=-1
        end
        else s:=s + str[j];
    end;
    if s <> '' then Next(False);

    Phrases:=TPhrases.Create;
    ExpandPhrases;
    For j:=0 to words.high do begin
      if words[j].Interval < 0 then ExpandPhrases;
      if vertical then (Phrases.Last as TVertPhrase).Add(words[j])
                  else (Phrases.Last as TPhrase).Add(words[j]);
      if words[j].IsHypenSign then Phrases.Last.WithOptionalHypen:=True;
      if words[j].EndsParagraph and (j < words.High) then ExpandPhrases;
    end;
    Words.Free;
  end;

  Constructor TBalon.Create(G: TAuGraph; ymin, xmin, ymax, xmax: integer);
  var x, y: integer;
  begin
    inherited Create;
    SetRect(ymax - ymin + 1, xmax - xmin + 1);
    For y:=ymin to ymax do
      For x:=xmin to xmax do
        if G[y][x] = 1 then Self[y - ymin][x - xmin]:=255
                       else Self[y - ymin][x - xmin]:= -1;
    MarkBorders;
  end;

  Destructor TBalon.Destroy;
  var j: integer;
  begin
    SetLength(Bmin, 0);
    SetLength(Bmax, 0);
    //if Assigned(Phrases) then For j:=0 to Phrases.High do SafeFree(Phrases[j]);
    SafeFree(Phrases);
    inherited;
  end;

  Procedure TBalon.MarkBorders;
  var
    x, y: integer;
  begin
{$rangechecks on}
    SetLength(Bmin, Height);
    SetLength(Bmax, Height);
    For x:=0 to 70 do SetLength(LoathingFactor[x], Height);
    _Bbeg:=Height;
    _Bend:=0;
    For y:=0 to Height - 1 do begin
      Bmin[y]:=Width;
      Bmax[y]:=0;
      For x:=0 to Width - 1 do
        if D[y][x] = 255 then begin
          Bmin[y]:=min(x, Bmin[y]);
          Bmax[y]:=max(x, Bmax[y]);
          _Bbeg:=min(_Bbeg, y);
          _Bend:=max(_Bend, y);
        end;
    end;
    for y:= _Bbeg + 1 to _Bend do
      if Bmin[y] > Bmax[y] then begin
        Bmin[y]:=Bmin[y-1];
        Bmax[y]:=Bmax[y-1];
      end;
    for y:= _Bbeg to _Bend do for x:=Bmin[y] to Bmax[y] do D[y][x]:=255;
{$rangechecks off}
  end;
  
  Procedure TVertBalon.MarkBorders;
  var
    x, y: integer;
  begin
{$rangechecks on}
    SetLength(Bmin, Width);
    SetLength(Bmax, Width);
    For y:=0 to 70 do SetLength(LoathingFactor[y], Width);
    For x:=0 to Width - 1 do begin
      Bmin[x]:=Height;
      Bmax[x]:=0;
      For y:=0 to Height - 1 do
        if D[y][x] = 255 then begin
          Bmin[x]:=min(y, Bmin[x]);
          Bmax[x]:=max(y, Bmax[x]);
          _Bbeg:=min(_Bbeg, x);
          _Bend:=max(_Bend, x);
        end;
    end;
    for x:= _Bbeg + 1 to _Bend do
      if Bmin[x] > Bmax[x] then begin
        Bmin[x]:=Bmin[x-1];
        Bmax[x]:=Bmax[x-1];
      end;
    for x:= _Bbeg to _Bend do for y:=Bmin[x] to Bmax[x] do D[y][x]:=255;
{$rangechecks off}
  end;
  
  Procedure THardcoreBalon.MarkBorders;
  var
    x, y: integer;
  begin
    SetLength(Bmin, Width);
    SetLength(Bmax, Width);
    For y:=0 to 70 do SetLength(LoathingFactor[y], Width);
    For x:=0 to Width - 1 do begin
      Bmin[x]:=Height;
      Bmax[x]:=0;
      For y:=0 to Height - 1 do
        if D[Height - 1 - y][x] = 255 then begin
          Bmin[x]:=min(y, Bmin[x]);
          Bmax[x]:=max(y, Bmax[x]);
          _Bbeg:=min(_Bbeg, x);
          _Bend:=max(_Bend, x);
        end;
    end;
    for x:= _Bbeg + 1 to _Bend do
      if Bmin[x] > Bmax[x] then begin
        Bmin[x]:=Bmin[x-1];
        Bmax[x]:=Bmax[x-1];
      end;
    for x:= _Bbeg to _Bend do for y:=Bmin[x] to Bmax[x] do D[Height - 1 - y][x]:=255;
  end;

  Procedure TAuMangaScan.FitBalon(s: string; F: TArrayOfFont;
     Charsize, mincharsize, Compression: integer; Hardcore: boolean;
     Color: integer; Comline: string; spacing: float; HypenLen: integer;
     CompressionLoathing,  ShortLinesLoathing,
     TransitionsLoathing, DeviationsLoathing: float; SmearCmd: string);
  var
    B: TBalon;
    y, x: integer;
  begin
    if Hardcore or (pos (TagMarker + TagHardcore, s) > 0)
    then B:=THardcoreBalon.Create(Koeff, ymin, xmin, ymax, xmax)
    else
      if (pos (TagMarker + TagVertical, s) > 0)
      then B:=TVertBalon.Create(Koeff, ymin, xmin, ymax, xmax)
      else B:=TBalon.Create(Koeff, ymin, xmin, ymax, xmax);
    B.balonNum:=baloncount;
    Try
      B.Put(s, F, Charsize, mincharsize, Compression, Zoom, spacing, HypenLen,
         CompressionLoathing,  ShortLinesLoathing,  TransitionsLoathing, DeviationsLoathing);
    Except
      For y:= 0 to B.Height - 1 do
        For x:= 0 to B.Width - 1 do
          if B[y][x] >= 0 then Koeff[y + ymin][x + xmin]:=1;
      ReportError(Koeff, 0, (ExceptObject as Exception).Message);
    End;
    PutBalon(B, Color, Comline, SmearCmd);
    B.Free;
  end;
  
  procedure TAuMangaScan.SmearHoles;
  begin
    if SmearPower = 0 then SmearMethod:= Sdraft;
    _SmearHoles(Self as TAuGraph, nil, nil, SmearMethod, SmearPower);
  end;
  
  procedure TAuColorMangaScan.SmearHoles;
  begin
    if SmearPower = 0 then SmearMethod:= Sdraft;
    _SmearHoles(Self as TAuGraph, GV, RV, SmearMethod, SmearPower);
  end;


  procedure TAuMangaScan.FillBalonBackground(color: integer; SmearCmd: string);
  var
    x, y: integer;
  begin
    if color < 0 then Exit; //color -1 means "transparent balon, don't fill".
    For y:=ymin to ymax do
      For x:=xmin to xmax do
        if Koeff[y][x] < MaxInt then begin
          Self[y][x]:=ColorToGray(color);
        end;
    //   -  .
  end;
  
  procedure TAuColorMangaScan.FillBalonBackground(color: integer;  SmearCmd: string);
  var
    x, y, i, dx, dy, scR, scG, scB, rad, prec: integer;

  begin
    if color < 0 then begin
      if SmearCmd <> '' then begin
        For y:=0 to Height - 1 do For x:=0 to Width - 1 do Alpha[y][x]:=0;
        TransparentCount:=0;
        i:=0;

//        if not (SmearMethod in [SDraft, SSimple, SHard, SCircular])
//           then SmearMethod:=SHard; //  
          //       ,
          //        .  .
          
        Repeat
          scR:=ColorToRed(StrToInt(StrParm(SmearCmd, i, [','])));
          scG:=ColorToGreen(StrToInt(StrParm(SmearCmd, i, [','])));
          scB:=ColorToBlue(StrToInt(StrParm(SmearCmd, i, [','])));
          prec:=StrToInt(StrParm(SmearCmd, i + 1, [',']));
          rad:=StrToInt(StrParm(SmearCmd, i + 2, [',']));
          For y:=ymin to ymax do
            For x:=xmin to xmax do
              if (abs(D[y][x] - scB)*0.2 + abs(RV[y][x] - scR)*0.3 + abs (GV[y][x] - scG)*0.5) < prec
                Then
                  For dy:=-rad to rad do
                    For dx:=-rad to rad do
                      if (x + dx >= 0) and (x + dx < Width) and (y + dy >=0) and (y + dy < Height)
                      and ((sqr(dx) + sqr(dy)) <= sqr(rad))
                        then begin
                          Alpha[y + dy][x + dx]:=255;
                          inc(TransparentCount);
                        end;
          inc(i, 3)
        until StrParm(SmearCmd, i, [',']) = '';
        SmearHoles;
      end;
      Exit; //color -1 means "transparent balon, don't fill".
    end;
    For y:=ymin to ymax do
      For x:=xmin to xmax do
        if Koeff[y][x] < MaxInt then begin
          Self[y][x]:=ColorToBlue(color);
          RV[y][x]:=ColorToRed(color);
          GV[y][x]:=ColorToGreen(color);
        end;
  end;


  Procedure TAuMangaScan.PutBalon(B: TBalon; Color: integer; Comline, SmearCmd: string);
  var
    x, y: integer;
  begin
    _NowWorkingWithBalons:=True;
    For y:=0 to Height - 1 do
      For x:=0 to Width - 1 do
        Koeff[y][x]:=MaxInt;
    For y:= 0 to B.Height - 1 do
 For x:= 0 to B.Width - 1 do
        if B[y][x] >= 0 then begin
          Koeff[y + ymin][x + xmin]:=Round((KoeffMultiplier / 255 ) *  B[y][x]);
          Red[y + ymin][x + xmin]:=0;
        end;
    FillBalonBackground(color, SmearCmd);
    For y:= 0 to B.Height - 1 do
      For x:= 0 to B.Width - 1 do
 if B[y][x] >= 0 then begin
          if Koeff[y + ymin][x + xmin] > Round((KoeffMultiplier / 255 ) *  250)
          then Koeff[y + ymin][x + xmin]:=maxInt;
        end;
    ProcessSFX(comline);
  end;

  procedure TAuGraph.Fill (v: integer);
  var jx, jy: integer;
  begin
    For jy:=0 to High do  For jx:=0 to D[0].High do D[jy][jx]:=v;
  end;

  procedure TAuGraph.Imbed (G: TAuGraph; y, x: integer);
  var jx, jy: integer;
  begin
    For jy:=0 to G.High do
      For jx:=0 to G[0].High do
        if (x + jx > D[0].High) or (y + jy > High)
          or (x + jx < 0) or (y + jy < 0) then Continue
        else Self[y + jy][x + jx]:=G[jy][jx];
  end;

  procedure TAuGraph.ImbedDarkest (G: TAuGraph; y, x: integer);
  var jx, jy: integer;
  begin
    For jy:=0 to G.High do
      For jx:=0 to G[0].High do
        if (x + jx > D[0].High) or (y + jy > High)
          or (x + jx < 0) or (y + jy < 0) then Continue
        else Self[y + jy][x + jx]:=min(Self[y + jy][x + jx], G[jy][jx]);
  end;

  Type TBorderArray = array[-3..34] of integer;

  Procedure BlurB(var Z: TBorderArray);
  var
    T: TBorderArray;
    j: integer;
  begin
    For j:=0 to 31 do
      T[j]:= min(min(min(Z[j-3],Z[j-2]),min(Z[j-1],Z[j])),min(min(Z[j+1],Z[j+2]),Z[j+3]));
    For j:=0 to 31 do Z[j]:=T[j];
  end;

  function MeasureInterleave(A, B: TAuGraph): integer;
  var
    L, R: TBorderArray;
    x, y, m: integer;
  begin
    For x:=-3 to 34 do begin L[x]:=31; R[x]:=31 end;
    // 
    For y:=0 to 31 do begin
      For x:=0 to A[0].High do begin
        R[y]:=x;
        if A[y][A[0].High - x] < 250 then Break;
      end;
      For x:=0 to B[0].High do begin
        L[y]:=x;
        if B[y][x] < 250 then Break;
      end;
    end;
    // 
    BlurB(L);
    // 
    m:=min(A[0].High, B[0].High);
    For y:=0 to 31 do m:=min(m, (L[y] + R[y]));
    Result:=m - 3;
  end;

  function MeasureVertInterleave(A, C: TAuGraph): integer;
  var
    T, B: TBorderArray;
    x, y, xr, m: integer;
  begin
    For x:=-3 to 34 do begin T[x]:=20; B[x]:=20 end;
    For x:=0 to 31 do begin
      For y:=0 to 31 do begin
        T[x]:=y;
        xr:=x + (A[0].High div 2) - 15;
        if (xr < 0) or (xr > A[0].High) then Continue;
        if A[A.High - y][xr] < 250 then Break;
      end;
      For y:=0 to 31 do begin
        B[x]:=y;
        xr:=x + (C[0].High div 2) - 15;
        if (xr < 0) or (xr > C[0].High) then Continue;
        if C[y][xr] < 250 then Break;
      end;
    end;
    // 
    BlurB(T);
    // 
    m:=20;
    For x:=0 to 31 do m:=min(m, (T[x] + B[x]));
    Result:=m - 3;
  end;

Const
  NotLetters: set of byte = [$20, $22, $27, $2A, $2D, $3D, $5F, $96,
                           $97, $AE, $AD, $AB, $8B, $9B, $BB,
                           $7E, $95, $B0, $7F, $88, $98, $A0];

  function TFont.Interval(a, b: char): integer;
  begin
    if not((ord(a) in NotLetters) or (ord(b) in NotLetters))
    and not IntervalsCalculated[ord(a), ord(b)]
    then begin
      Intervals[ord(a), ord(b)]:= - MeasureInterleave(Chars[ord(a)], Chars[ord(b)]);
      IntervalsCalculated[ord(a), ord(b)]:=true;
    end;
    if a in [NBSP, ' '] then Result:=NonBreakableSpaceWidth
                        else Result:=CharWidth[ord(a)] + Intervals[ord(a), ord(b)];
  end;
  
  function TFont.IntervalV(a, b: char): integer;
  begin
    if not IntervalsVCalculated[ord(a), ord(b)] and not ((a = NBSP) or (b = NBSP))
    then begin
      IntervalsV[ord(a), ord(b)]:= - MeasureVertInterleave(Chars[ord(a)], Chars[ord(b)]);
      IntervalsVCalculated[ord(a), ord(b)]:=true;
    end;
    if a in [NBSP, ' '] then Result:=NonBreakableSpaceWidth
                else Result:=32 {CharHeight[ord(a)]} + IntervalsV[ord(a), ord(b)];
  end;


  procedure TFont.Load;
  var
    G: TAuGraph;
    jx, jy, x, y, mup, mdn: integer;
  begin
    G:=TAuGraph.Create(_name);
    if (G.Width <> 512) or (G.Height <> 512) then YellAndDie(
      '      -'#10#13
     +'      512x512!'#10#13
     +'   ?..   "%s" - %dx%d!..',[ExtractFileName(name), G.Width, G.Height]);
    G._CleanJpegArtefacts(40, 0, 255);
    For jy:=2 to 15 do
      For jx:=0 to 15 do begin
        Chars[jx + jy*16]:= G.CutChar(jy * 32, jx *32, 32, 32);
        CharWidth[jx + jy*16]:=Chars[jx + jy*16][0].Length;
        CharHeight[jx + jy*16]:=32; //stub!
      end;
    G.Free;
    with Chars[ord(GuineaPigChar)] do begin
      mdn:= -1;
      For y:=0 to 31 do For x:=0 to Width - 1 do if D[y][x] < 200 then mdn:=y;
      For y:=31 downto 0 do For x:=0 to Width - 1 do if D[y][x] < 200 then mup:=y;
    end;
    if (mdn < 0) or ((mdn - mup) < 8) then YellAndDie(
      ' "%s"   "%s"  !'#10#13'-      ...',[GuineaPigChar, Name]);
    _zoom:= 32 / (mdn - mup + 1);
    {
    //    ,  
    //(  )
    For jy:=33 to 255 do
      For jx:=33 to 255 do begin
        IntervalsV[jy][jx]:= -MeasureVertInterleave(Chars[jy], Chars[jx]);
        IntervalsVCalculated[jy][jx]:=True;
        if (jx in NotLetters) or (jy in NotLetters) then Continue;
        Intervals[jy][jx]:= -MeasureInterleave(Chars[jy], Chars[jx]);
        IntervalsCalculated[jy][jx]:=True;
      end;}
    CharWidth[ord(NBSP)]:=Chars[ord(GuineaPigChar)].Width;
    CharHeight[ord(NBSP)]:=mdn - mup + 1;
    Chars[ord(NBSP)].SetRect(32, Chars[ord(GuineaPigChar)].Width);
    Chars[ord(NBSP)].TurnWhite;
  end;

  Constructor TFont.Create (Name: string);
  begin
    _name:=name;
  end;

  Function TFont.Line(s: string): TWord;
  var
    position: array of integer;
    j: integer;
  begin
    if s='' then s:=nbsp;
    SetLength(position, Length(s) + 1);
    position[1]:=0;
    For j:=2 to length(s) do
      Position[j]:=Position[j - 1] + Interval(s[j-1], s[j]);
    Result:=TWord.Create;
    Result.SetRect(32, Position[Length(s)] + CharWidth[ord(s[Length(s)])] + 1);
    Result.ZoomByFont:=Zoom;
    Result.Fill(255);
    For j:=1 to length(s) do
      Result.ImbedDarkest(Chars[ord(s[j])], 0, position[j]);
  end;
  
  Function TFont.LineLen(s: string): integer;
  var
    j, len: integer;
  begin
    if s='' then s:=nbsp;
    len:=0;
    For j:=2 to length(s) do inc(len, Interval(s[j-1], s[j]));
    Result:=len + CharWidth[ord(s[Length(s)])] + 1;
  end;
  
  Function TFont.VertLineLen(s: string): integer;
  var
    j, len: integer;
  begin
    if s='' then s:=nbsp;
    len:=0;
    For j:=2 to length(s) do inc(len, IntervalV(s[j-1], s[j]));
    Result:=len + CharHeight[ord(s[Length(s)])] + 1;
  end;


  function  TFont.VertLine(s: string): TWord;
  var
    position: array of integer;
    j, twd: integer;
  begin
    if s='' then s:=nbsp;
    SetLength(position, Length(s) + 1);
    position[1]:=0;
    twd:=2;
    For j:=2 to length(s) do
      Position[j]:=Position[j - 1] + IntervalV(s[j-1], s[j]);
    For j:=1 to length(s) do
      twd:=max(twd, CharWidth[ord(s[j])]);
    Result:=TWord.Create;
    Result.ZoomByFont:=Zoom;
    Result.SetRect(Position[Length(s)] + CharHeight[ord(s[Length(s)])] + 1, twd);
    Result.Fill(255);
    For j:=1 to length(s) do
      Result.ImbedDarkest(Chars[ord(s[j])], position[j], ((twd - CharWidth[ord(s[j])]) div 2));
  end;

  Destructor  TFont.Destroy;
  var c: integer;
  begin
    For c:=0 to 255 do if assigned (Chars[c]) then Chars[c].free;
    inherited;
  end;

  function TAuMangaScan.ScaleFactor(MinHeight, MinWidth, MaxHeight, MaxWidth: integer): float;
    function ToStr(i: integer): string;
    begin
      if i = MaxInt then Result:=''
                    else Result:=IntToStr(i);
    end;
    procedure yopt(f: float);
    begin
      YellAndDie(
'     (%d)   (%d)'#10#13+
'      (%s)  (%s)'#10#13+
' -     %d x %d.',
       [MinWidth, MinHeight, ToStr(MaxWidth), ToStr(MaxHeight), Round(f * EffectiveWidth), Round(f * EffectiveHeight)]);
    end;
  begin
    Result:=1;
    if (EffectiveHeight < MinHeight) or (EffectiveWidth < MinWidth)
    then begin
      if (EffectiveHeight < MinHeight) then zoom:=MinHeight / EffectiveHeight;
      if (EffectiveWidth < MinWidth) and ((MinWidth / EffectiveWidth) > Result)
        then Result:=MinWidth / EffectiveWidth;
      if (Round(Result * EffectiveWidth) > MaxWidth)
        or (Round(Result * EffectiveHeight) > MaxHeight) then Yopt(Result);
    end;
    if (EffectiveHeight > MaxHeight) or (EffectiveWidth > MaxWidth)
    then begin
      if (EffectiveWidth > MaxWidth) then Result:= MaxWidth / EffectiveWidth;
      if (EffectiveHeight > MaxHeight) and (Result > (MaxHeight / EffectiveHeight))
        then Result:= MaxHeight / EffectiveHeight;
      if (Round(Result * EffectiveWidth) < MinWidth)
        or (Round(Result * EffectiveHeight) < MinHeight) then Yopt(Result);
    end;
  end;
  
  Procedure TAuMangaScan._Upsample;
  var
    G: tAuGraph;
    y, x: integer;
  begin
    G:=Self.__Upsample;
    Self.SetRect(Height*2, Width*2);
    For y:=0 to height - 1 do
      For x:=0 to width - 1 do
        Self[y][x]:=G[y][x];
    G.Free;
    Koeff.SetRect(Height, Width);
    Red.Scale(Height, Width);
    Green.Scale(Height, Width);
    G:=Alpha.__Upsample;
    Alpha.Free;
    Alpha:=G;
  end;

  Procedure TAuColorMangaScan._Upsample;
  var
    G: tAuGraph;
  begin
    inherited;
    G:=RV.__Upsample;
    RV.Free;
    RV:=G;
    G:=GV.__Upsample;
    GV.Free;
    GV:=G;
  end;

  function TAuGraph.__Upsample: TAuGraph;
  var y, x: integer;
  begin
    Result:=TAuGraph.Create;
    Result.SetRect(Height * 2, Width * 2);
    For y:=0 to Result.Height - 1 do begin
      if (y div 300) <> ((y-1) div 300) then write('.');
      For x:=0 to Result.Width - 1 do
        Result[y][x]:=Self[y div 2][x div 2];
    end;
  end;

  Procedure TAuMangaScan.ApplySizeLimits(sc: float);
  begin
    zoom:=sc;
    if _cornersmarked then begin
      writeRu(format('(   %f..',[zoom]));
      Deform(Round(EffectiveHeight * zoom), Round(EffectiveWidth * zoom));
      writeRu('.).');
      _cornersmarked:=false;
    end
    else begin
      if abs (1 - zoom) > 0.0001 then begin
        writeRu(format('(  %f ..',[zoom]));
        Scale(Round(EffectiveHeight * zoom), Round(EffectiveWidth * zoom));
        writeRu('.).');
      end;
    end;
    EffectiveHeight:=Height;
    EffectiveWidth:=Width;
  end;

  Procedure TAuMangaScan.Upsample(sc: float);
  var zoom2: float;
  begin
    zoom:=sc;
    zoom2:=2 * zoom;
    if zoom2 * EffectiveHeight > MaxPicDimensions then zoom2:= MaxPicDimensions / EffectiveHeight;
    if zoom2 * EffectiveWidth > MaxPicDimensions then zoom2:= MaxPicDimensions / EffectiveWidth;
    zoom:=zoom2;
    if zoom < 1 then zoom:=1;
    if _cornersmarked then begin
      writeRu(format('(   %f..',[zoom2]));
      Deform(Round(EffectiveHeight * zoom), Round(EffectiveWidth * zoom2));
      writeRu('.).');
      _cornersmarked:=false;
    end
    else begin
      if abs (2 - zoom) > 0.001 then begin
        writeRu(format('(  %f ..',[zoom]));
        Scale(Round(EffectiveHeight * zoom), Round(EffectiveWidth * zoom));
        writeRu('.).');
      end
      else begin
        writeRu(format('( ..',[zoom]));
        _Upsample;
        writeRu('.).');
      end;
    end;
    EffectiveHeight:=Height;
    EffectiveWidth:=Width;
  end;
  
  Procedure HalfGr(Z: T2DAOI; HalfHeight, HalfWidth: boolean);
  var y, x: integer;
  begin
    if HalfHeight then begin
      for y:=0 to (Z.Length div 2) - 1 do for x:=0 to Z[0].High do Z[y][x]:=(Z[y*2][x] + Z[y*2 + 1][x]) div 2;
      Z.SetRect(Z.Length div 2, Z[0].Length);
    end;
    if HalfWidth then begin
      for y:=0 to Z.High do for x:=0 to (Z[0].Length div 2) - 1 do Z[y][x]:=(Z[y][x*2] + Z[y][x*2 + 1]) div 2;
      Z.SetRect(Z.Length, Z[0].Length div 2);
    end;
  end;

  Procedure ScaleGr(Z: T2DAOI; newHeight, NewWidth: integer);
  var
    N: T2DAOI;
    xm, ym, y, x, tx, ty, width, height: integer;
    fx, fy: float;
  begin
    While (Z.Length > NewHeight * 2.3) or (Z[0].Length > NewWidth * 2.3)
      do HalfGr (Z, Z.Length > NewHeight * 2.3, Z[0].Length > NewWidth * 2.3);
    N:=T2DAOI.Create;
    N.SetRect(NewHeight, NewWidth);
    width:=Z[0].Length;
    height:=Z.Length;
    For y:=0 to newHeight - 1 do begin
      ty:=Trunc(y * (height / NewHeight));
      fy:=frac(y * (height / NewHeight));
      For x:=0 to newWidth - 1 do begin
        tx:=Trunc(x * (width / NewWidth));
        fx:=frac(x * (width / NewWidth));
        N[y][x]:=Clamp(Round(
         (  Z[clamp(ty,0,height - 1)][clamp(tx,0,width - 1)] * sqr((1 - fy)*(1 - fx))
          + Z[clamp(ty,0,height - 1)][clamp(tx + 1,0,width - 1)] * sqr((1 - fy)*fx)
          + Z[clamp(ty + 1,0,height - 1)][clamp(tx,0,width - 1)] * sqr((fy * (1 - fx)))
          + Z[clamp(ty + 1,0,height - 1)][clamp(tx + 1,0,width - 1)] * sqr(fy *fx)
          ) / (sqr(fx * fy) + sqr((fx-1)*fy) + sqr((fy-1)*fx) + sqr((1 - fx)*(1 - fy)))), 0, 255);
      end;
    end;
    Z.SetRect(NewHeight, NewWidth);
    For y:=0 to Z.High do
      For x:=0 to Z[0].High do
        Z[y][x]:=N[y][x];
    N.Free;
  end;
  
  function TAuMangaScan._Deform(G: TAuGraph; NewHeight, NewWidth: integer): TAuGraph;
  var
    y, x: integer;
    yf, fy, fx, dx0v, dx1v, dy0v, dy1v,
    dxat, dxbt, dxab, dxbb, dyat, dybt, dyab, dybb: float;
  begin
    Result:=TAuGraph.Create;
    Result.SetRect(NewHeight, NewWidth);
    dx0v := (_corners[1, 0].x - _corners[0, 0].x) / (result.Height - 1);
    dy0v := (_corners[1, 0].y - _corners[0, 0].y) / (result.Height - 1);
    dx1v := (_corners[1, 1].x - _corners[0, 1].x) / (result.Height - 1);
    dy1v := (_corners[1, 1].y - _corners[0, 1].y) / (result.Height - 1);

    dxat := (_corners[0, 1].x - _corners[0, 0].x) / (result.Width div 2);
    dyat := (_corners[0, 1].y - _corners[0, 0].y) / (result.Width div 2);
    dxab := (_corners[1, 1].x - _corners[1, 0].x) / (result.Width div 2);
    dyab := (_corners[1, 1].y - _corners[1, 0].y) / (result.Width div 2);

    dxbt := (_corners[0, 2].x - _corners[0, 1].x) / (result.Width - 1 - result.Width div 2);
    dybt := (_corners[0, 2].y - _corners[0, 1].y) / (result.Width - 1 - result.Width div 2);
    dxbb := (_corners[1, 2].x - _corners[1, 1].x) / (result.Width - 1 - result.Width div 2);
    dybb := (_corners[1, 2].y - _corners[1, 1].y) / (result.Width - 1 - result.Width div 2);

//    if (NewHeight > G.Height) and (NewWidth > G.Width) then
      For y:=0 to result.Height - 1 do begin
        yf:=y / (result.Height - 1);
        if (y div 300) <> ((y - 1) div 300) then write('.');
        For x:=0 to result.Width div 2 - 1 do begin
          fy:= _corners[0, 0].y + dy0v * y + (1 - yf) * dyat * x + yf * dyab * x;
          fx:= _corners[0, 0].x + dx0v * y + (1 - yf) * dxat * x + yf * dxab * x;
          Result[y][x]:=G.{L}Inter(fy, fx);
        end;
        For x:=result.Width div 2 to result.Width - 1 do begin
          fy:= _corners[0, 1].y + dy1v * y + (1 - yf) * dybt * (x - result.Width div 2) + yf * dybb * (x - result.Width div 2);
          fx:= _corners[0, 1].x + dx1v * y + (1 - yf) * dxbt * (x - result.Width div 2) + yf * dxbb * (x - result.Width div 2);
          Result[y][x]:=G.{L}Inter(fy, fx);
        end
      end;
//    else;
  end;
  
  Procedure LScaleGr(Z: T2DAOI; newHeight, NewWidth: integer);
  var
    N: T2DAOI;
    xm, ym, y, x, tx, ty, width, height: integer;
    fx, fy: float;
  begin
    While (Z.Length > NewHeight * 2.3) or (Z[0].Length > NewWidth * 2.3)
      do HalfGr (Z, Z.Length > NewHeight * 2.3, Z[0].Length > NewWidth * 2.3);
    N:=T2DAOI.Create;
    N.SetRect(NewHeight, NewWidth);
    width:=Z[0].Length;
    height:=Z.Length;
    For y:=0 to newHeight - 1 do begin
      ty:=Trunc(y * (height / NewHeight));
      fy:=frac(y * (height / NewHeight));
      For x:=0 to newWidth - 1 do begin
        tx:=Trunc(x * (width / NewWidth));
        fx:=frac(x * (width / NewWidth));
        N[y][x]:=Clamp(Round(
         (  Z[clamp(ty,0,height - 1)][clamp(tx,0,width - 1)] * ((1 - fy)*(1 - fx))
          + Z[clamp(ty,0,height - 1)][clamp(tx + 1,0,width - 1)] * ((1 - fy)*fx)
          + Z[clamp(ty + 1,0,height - 1)][clamp(tx,0,width - 1)] * ((fy * (1 - fx)))
          + Z[clamp(ty + 1,0,height - 1)][clamp(tx + 1,0,width - 1)] * (fy *fx)
          ) / ((fx * fy) + ((fx-1)*fy) + ((fy-1)*fx) + ((1 - fx)*(1 - fy)))), 0, 255);
      end;
    end;
    Z.SetRect(NewHeight, NewWidth);
    For y:=0 to Z.High do
      For x:=0 to Z[0].High do
        Z[y][x]:=N[y][x];
    N.Free;
  end;

  
  function TAuGraph.HasAlpha: boolean;
  begin
    Result:=TransparentCount > 0;
  end;
  
  procedure TAuGraph.MarkCorners;
  begin
    //do nothing
  end;
  
  procedure TAuColorMangaScan.MarkCorners;
  begin
    //do nothing - only external markup is possible.
  end;
  
  Procedure TAuGraph.Scale(newHeight, NewWidth: integer);
  begin
    {if (newHeight > Height) and (newWidth > Width)
    then LScaleGr(Self as T2DAOI, newHeight, newWidth)
    else} ScaleGr(Self as T2DAOI, newHeight, newWidth);
  end;
  

  Procedure TAuMangaScan.Deform(newHeight, NewWidth: integer);
  var
    G: TAuGraph;
    y, x: integer;
  begin
    G:=Self._Deform(Self as TAuGraph, newHeight, NewWidth);
    Self. SetRect(newHeight, NewWidth);
    For y:=0 to Height - 1 do
      For x:=0 to width -1 do
        Self[y][x]:=G[y][x];
    G.Free;
    G:=Self._Deform(Red, newHeight, NewWidth);
    Red.Free;
    Red:=G;
    G:=Self._Deform(Green, newHeight, NewWidth);
    Green.Free;
    Green:=G;
    G:=Self._Deform(Alpha, newHeight, NewWidth);
    Alpha.Free;
    Alpha:=G;
    Koeff.SetRect(newHeight, newWidth);
    Edges.SetRect(newHeight, newWidth);
    _edgesdetected:=False;
  end;
  
  Procedure TAuColorMangaScan.Deform(newHeight, NewWidth: integer);
  var
    G: TAuGraph;
  begin
    inherited;
    G:=Self._Deform(RV, newHeight, NewWidth);
    RV.Free;
    RV:=G;
    G:=Self._Deform(GV, newHeight, NewWidth);
    GV.Free;
    GV:=G;
  end;



  Procedure TAuMangaScan.Scale(newHeight, NewWidth: integer);
  begin
    {if (newHeight > Height) and (newWidth > Width)
    then LScaleGr(Self as T2DAOI, newHeight, newWidth)
    else} ScaleGr(Self as T2DAOI, newHeight, newWidth);
    Red.Scale(newHeight, newWidth);
    Green.Scale(newHeight, newWidth);
//_SaveAsBmp('___koeff.bmp', Green);
    Alpha.Scale(newHeight, newWidth);
    Koeff.SetRect(newHeight, newWidth);
    Edges.SetRect(newHeight, newWidth);
    _edgesdetected:=False;
  end;
  
  Procedure TAuColorMangaScan.Scale(newHeight, NewWidth: integer);
  begin
    RV.Scale(newHeight, NewWidth);
    GV.Scale(newHeight, NewWidth);
    inherited Scale(newHeight, NewWidth);
  end;

  {$include aut_kambipng.inc}

  function TFont.Zoom: float;
  begin
    Result:=_zoom;
  end;
  
  function TAuGraph.CutChar(basey, basex, height, width: integer): TAuGraph;
  var
    x, y, maxX: integer;
  begin
    Result:=TAuGraph.Create;
    Result.SetRect(height, width);
    For y:=0 to Result.high do
      For x:=0 to Result[0].High do
        Result[y][x]:=Self[y + basey][x + basex];
    MaxX:=0;
    For x:= 0 to Result[0].High do
      For y:=0 to Result.high do
        if Result[y][x] < 250 then MaxX:=max(maxX, x);
    Result.SetRect(height, MaxX + 1);
  end;

  procedure TAuMangaScan.ProcessSFX(comline: string);
  var
    L, B: integer;
    R, A: float;
    Tex: TAOS;
    Rad, Atk: TAOI;
  begin
    Tex:=TAOS.Create;
    Rad:=TAOI.Create;
    Atk:=TAOI.Create;
    L:=0;
    R:=0;
    While StrParm(comline, L*3, [',']) <> ''
    do begin
      Tex.Add(StrParm(comline, L*3, [',']));
      A:=StrToFloat(StrParm(comline, L*3 + 2, [',']));
      R:=R + zoom * (Trunc(A) + StrToFloat( StrParm(comline, L*3 + 1, [','])));
      A:=sqrt(zoom) * A; //    
      Rad.Add(Round(R * 100));
      Atk.Add(Round(A * 100));
      inc(L);
    end;
    MarkupSFX(trunc(R + 2));
    if Rad[0] < 0 then ExpandSFXMarkupToNegativeSide;
    For L:=Tex.High downto 0 do begin
      if Tex[L][1]='%' then begin
        B:=StrToInt(StrDelSet(Tex[L], ['%']));
        DrawSFX(Rad[L] /100, Atk[L] / 100, B)
      end
      else DrawSFX(Rad[L] / 100, Atk[L] / 100, ExtractFilePath(_filename) + Tex[L]);
    end;
    Tex.Free;
    Rad.Free;
    Atk.Free;
  end;

  procedure TAuGraph._CleanJPEGArtefacts (Power, Brmin, Brmax: integer);
  var y, x: integer;
  begin
    For y:=1 to High-2 do
      For x:=1 to D[0].High - 2 do begin
        if (D[y][x] < Power + Brmin)
        and not ((D [y][x-1] > Power + Brmin) or (D[y][x+1] > Power + Brmin)
                 or (D[y-1][x] > Power + Brmin) or (D[y+1][x] > Power + Brmin))
          then D[y][x]:=Brmin;
        if (D[y][x] > (Brmax - Power))
        and not ((D[y][x-1] < (Brmax - Power)) or (D[y][x+1] < (Brmax - Power))
                 or (D[y-1][x]  < (Brmax - Power)) or (D[y+1][x]  < (Brmax - Power)))
          then D[y][x]:=Brmax;
      end;
  end;
  
  procedure TAuGraph._FindLevels;
  var y, x: integer;
  begin
    MinBrightness:=255;
    MaxBrightness:=0;
    For y:= 0 to Height - 1 do
      For x:=0 to Width - 1 do begin
        MinBrightness:=min(MinBrightness, D[y][x]);
        MaxBrightness:=max(MaxBrightness, D[y][x]);
      end;
  end;
  
  procedure TAuMangaScan.AutoClean(Power: integer);
  var
    y, x, m, w: integer;
    mul: float;
    weight: array[0..255] of integer;
  begin
    _FindLevels;
    _CleanJpegArtefacts(Power, MinBrightness, MaxBrightness);
    For x:=0 to 255 do weight[x]:=0;
    For y:= 0 to Height - 1 do
      For x:=0 to Width - 1 do
        inc(weight[clamp(D[y][x], 0, 255)]);
    m:=0;
    w:=0;
    repeat
      inc(w, weight[m]);
      inc(m);
    Until (m >= (power div 2)) or (w > (Width * Height * Power) div 20);
    MinBrightness:=max(m, Power);
    m:=255;
    w:=0;
    repeat
      inc(w, weight[m]);
      dec(m);
    Until (m <= 255 - (power div 2)) or (w > (Width * Height * Power) div 20);
    MaxBrightness:=min(m, 255 - Power);
    mul:=255 / (MaxBrightness - MinBrightness);
    For y:= 0 to Height - 1 do
      For x:=0 to Width - 1 do
        D[y][x]:=clamp(Round((D[y][x] - MinBrightness) * mul), 0, 255);
    MinBrightness:=0;
    MaxBrightness:=255;
  end;

  procedure TAuColorMangaScan.AutoClean (Power: integer);
  var
    y, x: integer;
    mul: float;
  begin
    if Power = 0 then Exit;
    Self._FindLevels;
    _CleanJPEGArtefacts (Power, MinBrightness, MaxBrightness);
    RV._FindLevels;
    RV._CleanJPEGArtefacts (Power, RV.MinBrightness, RV.MaxBrightness);
    GV._FindLevels;
    GV._CleanJPEGArtefacts (Power, GV.MinBrightness, GV.MaxBrightness);

    MaxBrightness:=max(255 - Power, min(MaxBrightness, min(RV.MaxBrightness, GV.MaxBrightness)));
    MinBrightness:=min(Power, max(MinBrightness, max(RV.MinBrightness, GV.MinBrightness)));
    mul:=255 / (MaxBrightness - MinBrightness);
    For y:= 0 to Height - 1 do
      For x:=0 to Width - 1 do begin
        D[y][x]:=Clamp(Round((D[y][x] - MinBrightness) * mul), 0, 255);
        RV[y][x]:=Clamp(Round((RV[y][x] - MinBrightness) * mul), 0, 255);
        GV[y][x]:=Clamp(Round((GV[y][x] - MinBrightness) * mul), 0, 255);
      end;
    MinBrightness:=0;
    MaxBrightness:=255;
  end;
  

  procedure TAuMangaScan._DrawSFX(G: TAuGraph; diameter, blur: float; brightness: integer);
  var
    x, y: integer;
    opacity, attack: float;
  begin
    if blur < 0.01 then blur:=0.01;
    attack:=1 / blur;
    For y:=ymin to ymax do
      For x:=xmin to xmax do begin
        if Koeff[y][x] < ((diameter + 1) * KoeffMultiplier) then begin
          opacity:= (diameter + blur - (Koeff[y][x] / KoeffMultiplier)) * attack;
          if opacity < 0.0 then opacity:=0.0 else Green[y][x]:=0;
          if opacity > 1.0 then opacity:=1.0;
          G[y][x]:=clamp( round( (1 - opacity ) * G[y][x] + opacity * brightness ),0 , 255);
        end;
      end;
  end;
  
  procedure TAuMangaScan.DrawSFX(diameter, blur: float; brightness: integer);
  begin
    _DrawSFX(Self as TAuGraph, diameter, blur, ColorToGray(brightness));
  end;
  
  procedure TAuColorMangaScan.DrawSFX(diameter, blur: float; brightness: integer);
  begin
    _DrawSFX(Self as TAuGraph, diameter, blur, ColorToBlue(brightness));
    _DrawSFX(GV, diameter, blur, ColorToGreen(brightness));
    _DrawSFX(RV, diameter, blur, ColorToRed(brightness));
  end;


  function Saturate (Value, Range: integer): integer;
  begin
    Result:=Value - ((Value) div Range) * Range;
  end;

  procedure TAuMangaScan.DrawSFX(diameter, blur: float; texturefile: string);
  var
    x, y: integer;
    opacity, attack: float;
    tex: TAuTexture;
  begin
    if blur < 0.01 then blur:=0.01;
    attack:=1 / blur;
    tex:=TAuTexture.Create(texturefile);
    For y:=ymin to ymax do
      For x:=xmin to xmax do begin
        if Koeff[y][x] < ((diameter + 2) * KoeffMultiplier) then begin
          opacity:= (1.0 * diameter + blur - (Koeff[y][x] / KoeffMultiplier)) * attack;
          if opacity < 0.0 then opacity:=0.0 else Green[y][x]:=0;
          if opacity > 1.0 then opacity:=1.0;
          D[y][x]:=clamp( round( (1 - opacity ) * D[y][x] + opacity * tex.Texel((y - ymin)/zoom, (x - xmin)/zoom)),0 , 255);
          Green[y][x]:=0;
        end;
      end;
    tex.Free;
  end;
  
  procedure TAuColorMangaScan.DrawSFX(diameter, blur: float; texturefile: string);
  var
    x, y: integer;
    opacity, attack: float;
    tex: TAuColorTexture;
  begin
    if blur < 0.01 then blur:=0.01;
    attack:=1 / blur;
    tex:=TAuColorTexture.Create(texturefile);
    For y:=ymin to ymax do
      For x:=xmin to xmax do begin
        if Koeff[y][x] < ((diameter + 2) * KoeffMultiplier) then begin
          opacity:= (1.0 * diameter + blur - (Koeff[y][x] / KoeffMultiplier)) * attack;
          if opacity < 0.0 then opacity:=0.0 else Green[y][x]:=0;
          if opacity > 1.0 then opacity:=1.0;
          D[y][x]:=clamp( round( (1 - opacity ) * D[y][x] + opacity * tex.TexelB((y - ymin)/zoom, (x - xmin)/zoom)),0 , 255);
          RV[y][x]:=clamp( round( (1 - opacity ) * RV[y][x] + opacity * tex.TexelR((y - ymin)/zoom, (x - xmin)/zoom)),0 , 255);
          GV[y][x]:=clamp( round( (1 - opacity ) * GV[y][x] + opacity * tex.TexelG((y - ymin)/zoom, (x - xmin)/zoom)),0 , 255);
        end;
      end;
    tex.Free;
  end;

  procedure TAuMangaScan.ExpandSFXMarkup(diameter: integer);
  var
    i, j, delta, y, x, p, n: integer;
  begin
    if diameter < sfx_diameter_marked then Exit;
    delta:= diameter - sfx_diameter_marked;
    dec(xmin, delta); xmin:=max(0, xmin);
    inc(xmax, delta); xmax:=min(D[0].High, xmax);
    dec(ymin, delta); ymin:=max(0, ymin);
    inc(ymax, delta); ymax:=min(High, ymax);
    For y:=ymin to ymax do
      For x:=xmin to xmax do begin
        p:=Koeff[y][x];
        if p > KoeffMultiplier then Continue;
        For i:=max(0, y - diameter) to min(High, y + diameter) do
          For j:= max(0, x - diameter) to min(D[0].High, x + diameter) do begin
            if Koeff[i][j] < KoeffMultiplier then Continue;
            n:=round(p + KoeffMultiplier * sqrt( sqr( i - y ) + sqr( j - x ) ) );
            Koeff[i][j]:= min (n, Koeff[i][j]);
          end;
      end;
    sfx_diameter_marked:=diameter;
    write('.');
  end;

  procedure TAuMangaScan.ExpandSFXMarkupToNegativeSide;
  var
    x, y, v, i, j, r, rmax: integer;
    Hit: boolean;
    function Probe(py, px: integer): integer;
    begin
      Result:= -MaxProbableSFXThickness * KoeffMultiplier;
      if (py < ymin) or (py > ymax) or (px < xmin) or (px > xmax) then Exit;
      if Koeff[py][px] < KoeffMultiplier then Exit;
      if Koeff[py][px] > (KoeffMultiplier * 2) then Exit;
      if not Hit then rmax:=round( 1.6 * sqrt( sqr( py - y) + sqr( px - x)));
      Hit:=True;
      Result:=Koeff[py][px] - round( KoeffMultiplier * (Sqrt( sqr( py - y) + sqr (px - x))))
    end;
  begin
    For y:=ymin to ymax do begin
      For x:=xmin to xmax do begin
        if Koeff[y][x] > KoeffMultiplier then Continue;
        r:=0;
        rmax:=MaxProbableSFXThickness;
        v:= -MaxProbableSFXThickness * KoeffMultiplier;
        Hit:=false;
        Repeat
          inc(r);
          For j:=0 to r do
            v:=max(v, max(max(Probe(y - j, x + r -j), Probe(y - j, x - r + j)),
                          max(Probe(y + j, x + r -j), Probe(y + j, x - r + j)) ));
        until r >= rmax;
        Koeff[y][x]:=min(Koeff[y][x], v);
      end;
    end;
    write('.');
  end;

  procedure TAuMangaScan.Transform;
  var
    x, y, R, G, B, A: integer;
    T: float;
  begin
//...');
    Green:= TAuGraph.Create;
    Red:= TAuGraph.Create;
    Alpha:= TAuGraph.Create;
    Koeff:= TAuGraph.Create;
    Edges:= TAuGraph.Create;
    Green.SetRect(Height, Width);
    Red.SetRect(Height, Width);
    Alpha.SetRect(Height, Width);
    Koeff.SetRect(Height, Width);
    Edges.SetRect(Height, Width);
    Green.ClearMem;
    Red.ClearMem;
    Alpha.ClearMem;
    Koeff.ClearMem;
    For y:=0 to High do begin
      For x:=0 to D[0].High do begin
        B:= (D[y][x] shr 16)and $000000FF;
        G:= (D[y][x] shr 8) and $000000FF;
        R:= (D[y][x]) and $000000FF;
        A:= (D[y][x] shr 24) and $000000FF;
        Green[y][x]:=round(A / 255 * max(0, G - max(R, B)));
        Red[y][x]:=round(A / 255 * max(0, R - max(G, B)));
        Alpha[y][x]:=255 - A;
        T:=max(1, B) / max(1, B + Green[y][x]);
        D[y][x]:=Round(B / T);
        if Alpha[y][x] > MinAlphaToSmear then inc(TransparentCount);
      end;
    end;
    MinBrightness:=0;
    MaxBrightness:=255;
    LoadMarkupExternal;
    zoom:=1.0;
    EffectiveWidth:=Width;
    EffectiveHeight:=Height;
    WriteRu('..');
  end;
  
  
  procedure TAuColorMangaScan.Transform;
  var
    x, y: integer;
  begin
//...');
    Green:= TAuGraph.Create;
    Red:= TAuGraph.Create;
    Alpha:= TAuGraph.Create;
    Koeff:= TAuGraph.Create;
    Edges:= TAuGraph.Create;
    RV:=TAuGraph.Create;
    GV:=TAuGraph.Create;

    RV.SetRect(Height, Width);
    GV.SetRect(Height, Width);
    Alpha.SetRect(Height, Width);
    Red.SetRect(Height, Width);
    Red.ClearMem;
    Koeff.SetRect(Height, Width);
    Green.SetRect(Height, Width);
    Green.ClearMem;
    Edges.SetRect(Height, Width);
    For y:=0 to Height - 1 do
      For x:=0 to Width - 1 do begin
        Alpha[y][x]:= 255 - ((D[y][x] shr 24) and $000000FF);
        RV[y][x]:=D[y][x] and $000000FF;
        GV[y][x]:=(D[y][x] shr 8) and $000000FF;
        D[y][x]:=(D[y][x] shr 16) and $000000FF;
        if Alpha[y][x] > MinAlphaToSmear then inc(TransparentCount);
      end;
    LoadMarkupExternal;
    zoom:=1.0;
    MinBrightness:=0;
    MaxBrightness:=255;
    EffectiveWidth:=Width;
    EffectiveHeight:=Height;
    WriteRu('..');
  end;

  procedure TAuMangaScan.BalonMarkupRecursiveFill (x, y: integer);
  begin
    if (x < 0) or (y < 0) or (x > D[0].High) or (y > High)
      or (Red[y][x] < MinMarkupTreshold) or (Koeff[y][x] = 1)
      or ((recursion > 1) and (Red[y][x] = sfxbasevalue))
        then Exit;
    Koeff[y][x]:=1;
    inc(recursion);// WriteLn(recursion);
    if Recursion > RecursionLimit then ReportError(Koeff, 0, format(
      '       %d  .'#10#13
      +' ,   ""  .', [baloncount]));
     if y < ymin then ymin:=y;
     if y > ymax then ymax:=y;
     if x < xmin then xmin:=x;
     if x > xmax then xmax:=x;
    BalonMarkupRecursiveFill(x-1, y);
    BalonMarkupRecursiveFill(x+1, y);
    BalonMarkupRecursiveFill(x, y-1);
    BalonMarkupRecursiveFill(x, y+1);
    dec(recursion);
  end;

  procedure TAuMangaScan.MarkupBalon;
  var
    Weight: array[byte] of integer;
    x, y, v, j: integer;
  begin
    inc(baloncount);
    Koeff.ClearMem;
    FillChar(Weight, SizeOf(Weight), 0);
    For y:=0 to High do
      For x:=0 to D[0].High do
        Inc(Weight[Red[y][x]]);
    sfxbasevalue:=255;
    While Weight[sfxbasevalue] < SFXWeightTreshold do begin
      dec(sfxbasevalue);
      if sfxbasevalue < MinimumMarkupBrightness then ReportError(red, MinMarkupTreshold, format(
' %d  ,   .'#10#13+
'      %d.'#10#13+
' ,     (   '#10#13+
'    ,    '#10#13+
'    )' ,[baloncount, MinimumMarkupBrightness]));
    end;
    ymin:=High; ymax:=0; xmin:=D[0].High; xmax:=0;
    recursion:=0;
    For y:=0 to High do
      For x:=0 to D[0].High do
        if Red[y][x] = sfxbasevalue then BalonMarkupRecursiveFill(x, y);
  end;

  procedure TAuMangaScan.SFXMarkupRecursiveFill (x, y: integer);
  begin
    if (x < 0) or (y < 0) or (x > D[0].High) or (y > High)
      or (Green[y][x] < MinMarkupTreshold) or (Koeff[y][x] < MaxInt)
      or ((recursion > 1) and (Green[y][x] > sfxbasevalue - SfxWeightDiff)) then Exit;
    Koeff[y][x]:=Round ( KoeffMultiplier * ( 1.0 - (Green[y][x] / sfxbasevalue) ) );
    Green[y][x]:=0;
    inc(recursion);
    if Recursion > RecursionLimit then ReportError(Green, MinMarkupTreshold, format(
'       %d  .'#10#13+
' ,   ""      .'#10#13+
'    ,     .' ,[sfxcount]));
     if y < ymin then ymin:=y;
     if y > ymax then ymax:=y;
     if x < xmin then xmin:=x;
     if x > xmax then xmax:=x;
//writeln(recursion);
    SFXMarkupRecursiveFill(x-1, y);
    SFXMarkupRecursiveFill(x, y-1);
    SFXMarkupRecursiveFill(x+1, y);
    SFXMarkupRecursiveFill(x, y+1);
    dec(recursion);
  end;

  procedure TAuMangaScan.MarkupSFX(diameter: integer);
  var
    Weight: array[byte] of integer;
    x, y, v, j: integer;
  begin
    inc(sfxcount);
    if not (_NowWorkingWithBalons) then begin
      FillChar(Weight, SizeOf(Weight), 0);
      For y:=0 to High do
        For x:=0 to D[0].High do
          Inc(Weight[Green[y][x]]);
      sfxbasevalue:=255;
      While Weight[sfxbasevalue] < SFXWeightTreshold do begin
        dec(sfxbasevalue);
        if sfxbasevalue < MinimumMarkupBrightness then begin
          ReportError(Green, MinMarkupTreshold, format(
' %d  ,   .'#10#13+
'      %d.'#10#13+
' ,        '#10#13+
' (       ,'#10#13+
'         )', [sfxcount, MinimumMarkupBrightness]));
        end;
      end;

      For y:=0 to High do For x:=0 to D[0].High do Koeff[y][x]:=MaxInt;
      ymin:=High; ymax:=0; xmin:=D[0].High; xmax:=0;
      recursion:=0;
      For y:=0 to High do
        For x:=0 to D[0].High do
          if Green[y][x] > max(sfxbasevalue - SfxWeightDiff, MinimumMarkupBrightness)
            then SFXMarkupRecursiveFill(x, y);
      sfx_diameter_marked:=0;
  //_SaveAsBmp(ExtractFilePath(ParamStr(0))+'_testKoeff.bmp', Koeff);
    end;
    ExpandSFXMarkup(diameter);
  end;

  destructor TAuMangaScan.Destroy;
  begin
    SafeFree(Green);
    SafeFree(Alpha);
    SafeFree(Koeff);
    SafeFree(Red);
    inherited;
  end;
  
  destructor TAuColorMangaScan.Destroy;
  begin
    SafeFree(RV);
    SafeFree(GV);
    inherited;
  end;

  Constructor TAuGraph.Create;
  begin
    inherited;
    _FileName:='';
  end;

  Constructor TAuGraph.Create(fileName: string);
  begin
    inherited Create;
    _filename:=filename;
    if (FileExists(ChangeFileExt(filename, '.png'))) and not LibPngDisabled
      then _ReadFromPng(filename)
      else
        if FileExists(ChangeFileExt(filename, '.tga'))
        then _ReadFromTga(filename)
        else
          if LibPngDisabled
            then YellAndDie('  !'#10#13'("%s")', [ChangeFileExt(filename, '.tga')])
            else YellAndDie('  !'#10#13'("%s")', [ChangeFileExt(filename, '.png')]);
    MarkCorners;
    Transform;
  end;
  
  function TAuGraph.Clone: TAuGraph;
  var y, x: integer;
  begin
    Result:=TAuGraph.Create;
    Result.SetRect(Height, Width);
    For y:=0 to Height - 1 do
      For x:=0 to Width - 1 do
        Result[y][x]:=Self[y][x];
  end;
  
  function TAuColorMangaScan.Clone: TAuGraph;
  var y, x: integer;
  begin
    Result:=TAuGraph.Create;
    Result.SetRect(Height, Width);
    For y:=0 to Height - 1 do
      For x:=0 to Width - 1 do
        Result[y][x]:=Round(Self[y][x] * 0.2 + RV[y][x] * 0.3 + GV[y][x] * 0.5);
  end;

  Procedure TAuGraph._ReadFromTGA(Filename: string);
  const
    TGAheader : array [0..11] of byte = (0,0,2,0,0,0,0,0,0,0,0,0);	// Uncompressed TGA Header
    TGAComheader : array [0..11] of byte = (0,0,10,0,0,0,0,0,0,0,0,0); // Compressed TGA Header
    Compressed: boolean = False;
  var
    W, H, x, y: integer;
    TGAcompare : array [0..11] of byte;// Used To Compare TGA Header
    header : array [0..5] of byte;// First 6 Useful Bytes From The Header
    bytesPerPixel : integer;// Holds Number Of Bytes Per Pixel Used In The TGA File
    imageSize : integer;	   // Used To Store The Image Size When Setting Aside Ram
    i : integer;		   // Temporary Variable

    Tm: Char;
    tgafile : TFileStream;

    PixelCount, CurrentPixel, CurrentByte: integer;
    ColorBuffer: Pchar;
    ChunkHeader: byte;
    Counter, Ret: integer;
    buffer: array of byte;
  begin
    tgafile := TFileStream.Create(ChangeFileExt(filename, '.tga'), fmOpenRead + fmShareDenyWrite);
    Ret := tgafile.Read(TGAcompare, sizeof(TGAcompare));

    if CompareMem(@TGAheader, @TGAcompare, sizeof(TGAheader)) // Does The Header Match What We Want?
      then Compressed := False
    else begin
      if CompareMem(@TGAComheader, @TGAcompare, sizeof(TGAComheader)) // Does The Header Match What We Want?
        then Compressed := True
        else YellAndDie(' ,   TGA  .   .'#10#13'(%s)'#10#13'  24- 32-  TGA (  ,   )', [filename]);
    end;
    tgafile.Read(header, sizeof(header));
    W:= header[1] * 256 + header[0]; // Determine The TGA Width	(highbyte*256+lowbyte)
    H:= header[3] * 256 + header[2]; // Determine The TGA Height	(highbyte*256+lowbyte)

    if (W <= 0) or (H <= 0) or ((header[4] <> 24) and (header[4] <> 32))
      then YellAndDie('  TGA .   .'#10#13'(%s)'#10#13'  24- 32-  TGA (  ,   )', [filename]);

    bytesPerPixel:= header[4] div 8;// Grab The TGA's Bits Per Pixel (24 or 32) & Divide By 8 To Get The Bytes Per Pixel
    imageSize:= W * H * bytesPerPixel;	// Calculate The Memory Required For The TGA Data
    if (H > MaxPicDimensions) or (W > MaxPicDimensions) then YellAndDie(
      '    - %dx%d,'#10#13
      +'    ?.. %dx%d!'#10#13'( "%s")', [MaxPicDimensions, W, H, filename]);
    SetRect(H, W);
    if Not Compressed then begin
      SetLength(buffer, imageSize);// Reserve Memory To Hold The TGA Data
      tgafile.Read(buffer[0], integer(imageSize));
      For y:=0 to H-1 do
        For x:=0 to W-1 do
          D [H - y - 1][x]:=(integer(buffer[(y*W + x)*3 + 2]) + buffer[(y*W + x)*3 + 1] shl 8 + buffer[(y*W + x)*3] shl 16) or $FF000000;
    end
    else begin //COMPRESSED TGA'S
      PixelCount := W * H;
      CurrentPixel := 0;
      x:=0;
      y:=H-1;
      CurrentByte := 0;
      GetMem(ColorBuffer, BytesPerPixel);
      Repeat
        ChunkHeader := 0;
        tgafile.Read(ChunkHeader, sizeof(byte));
        if ChunkHeader < 128 then begin
          ChunkHeader := ChunkHeader + 1;
          For Counter := 0 to ChunkHeader-1 do begin
            tgafile.Read(ColorBuffer^, BytesPerPixel);
            D[y][x]:=integer(ColorBuffer[2]) + integer(ColorBuffer[1]) shl 8 + integer(ColorBuffer[0]) shl 16;
            if BytesPerPixel = 4
              then  D[y][x]:=D[y][x] + integer(ColorBuffer[3]) shl 24
              else  D[y][x]:=D[y][x] or $FF000000;
            CurrentByte := CurrentByte + bytesPerPixel;
            inc(CurrentPixel);
            inc(x);
            if x = W then begin
              x:=0;
              dec(y);
            end;
            if CurrentPixel > PixelCount
              then YellAndDie('TGA  .   . (%s)', [filename]);
          end;
        end
        else begin//Chunkheader > 128
          ChunkHeader := ChunkHeader - 128;
          tgafile.Read(ColorBuffer^, BytesPerPixel);
          For Counter := 0 to ChunkHeader do begin
            D[y][x]:=integer(ColorBuffer[2]) + integer(ColorBuffer[1]) shl 8 + integer(ColorBuffer[0]) shl 16;
            if BytesPerPixel = 4
              then  D[y][x]:=D[y][x] + integer(ColorBuffer[3]) shl 24
              else  D[y][x]:=D[y][x] or $FF000000;
            CurrentByte := CurrentByte + bytesPerPixel;
            inc(CurrentPixel);
            inc(x);
            if x = W then begin
              x:=0;
              dec(y);
            end;
            if CurrentPixel > PixelCount
              then YellAndDie('TGA  .   .  (%s)',[filename]);
          end;
        end;
      Until CurrentPixel >= PixelCount;
    end;
    tgafile.Free;
  end;

  Destructor TAuGraph.Destroy;
  begin
    _filename:='';
    inherited;
  end;
  
  function TAuGraph._readwidth: integer;
  begin
    Result:=D[0].Length;
  end;

  function TAuTexture._Ixel(y, x: integer): integer;
  begin
    Result:=D[Saturate(y, High)][Saturate(x, D[0].High)];
  end;

  function TAuColorTexture._Ixel(y, x: integer): integer;
  begin
    Result:=Round(_IxelR(y, x) * 0.3 + _IxelG(y, x) * 0.5 + _IxelB(y, x) * 0.2);
  end;

  function TAuTexture.Texel(y, x: float): integer;
  var
    fy, fx, val, weight: float;
    iy, ix: integer;
  begin
    iy:=Trunc(y);
    ix:=Trunc(x);
    fy:=Frac(y);
    fx:=Frac(x);
    val:= sqr((1 - fy)*(1 - fx)) * _ixel(iy,     ix)
        + sqr((1 - fy)*   fx   ) * _ixel(iy,     ix + 1)
        + sqr(   fy   *   fx   ) * _ixel(iy + 1, ix + 1)
        + sqr(   fy   *(1 - fx)) * _ixel(iy + 1, ix);
    weight:=sqr((1 - fy)*(1 - fx))
          + sqr((1 - fy)*   fx   )
          + sqr(   fy   *   fx   )
          + sqr(   fy   *(1 - fx));
    Result:= round( val / weight);
  end;
  
  function TAuColorTexture.TexelB(y, x: float): integer;
  var
    fy, fx, val, weight: float;
    iy, ix: integer;
  begin
    iy:=Trunc(y);
    ix:=Trunc(x);
    fy:=Frac(y);
    fx:=Frac(x);
    val:= sqr((1 - fy)*(1 - fx)) * _ixelB(iy,     ix)
        + sqr((1 - fy)*   fx   ) * _ixelB(iy,     ix + 1)
        + sqr(   fy   *   fx   ) * _ixelB(iy + 1, ix + 1)
        + sqr(   fy   *(1 - fx)) * _ixelB(iy + 1, ix);
    weight:=sqr((1 - fy)*(1 - fx))
          + sqr((1 - fy)*   fx   )
          + sqr(   fy   *   fx   )
          + sqr(   fy   *(1 - fx));
    Result:= round( val / weight);
  end;

  function TAuColorTexture.TexelG(y, x: float): integer;
  var
    fy, fx, val, weight: float;
    iy, ix: integer;
  begin
    iy:=Trunc(y);
    ix:=Trunc(x);
    fy:=Frac(y);
    fx:=Frac(x);
    val:= sqr((1 - fy)*(1 - fx)) * _ixelG(iy,     ix)
        + sqr((1 - fy)*   fx   ) * _ixelG(iy,     ix + 1)
        + sqr(   fy   *   fx   ) * _ixelG(iy + 1, ix + 1)
        + sqr(   fy   *(1 - fx)) * _ixelG(iy + 1, ix);
    weight:=sqr((1 - fy)*(1 - fx))
          + sqr((1 - fy)*   fx   )
          + sqr(   fy   *   fx   )
          + sqr(   fy   *(1 - fx));
    Result:= round( val / weight);
  end;
  
  function TAuColorTexture.TexelR(y, x: float): integer;
  var
    fy, fx, val, weight: float;
    iy, ix: integer;
  begin
    iy:=Trunc(y);
    ix:=Trunc(x);
    fy:=Frac(y);
    fx:=Frac(x);
    val:= sqr((1 - fy)*(1 - fx)) * _ixelR(iy,     ix)
        + sqr((1 - fy)*   fx   ) * _ixelR(iy,     ix + 1)
        + sqr(   fy   *   fx   ) * _ixelR(iy + 1, ix + 1)
        + sqr(   fy   *(1 - fx)) * _ixelR(iy + 1, ix);
    weight:=sqr((1 - fy)*(1 - fx))
          + sqr((1 - fy)*   fx   )
          + sqr(   fy   *   fx   )
          + sqr(   fy   *(1 - fx));
    Result:= round( val / weight);
  end;

  function TAuColorTexture._IxelB(y, x: integer): integer;
  begin
    Result:=$FF and ((D[Saturate(y, High)][Saturate(x, D[0].High)]) shr 16);
  end;

  function TAuColorTexture._IxelG(y, x: integer): integer;
  begin
    Result:=$FF and ((D[Saturate(y, High)][Saturate(x, D[0].High)]) shr 8);
  end;

  function TAuColorTexture._IxelR(y, x: integer): integer;
  begin
    Result:=$FF and D[Saturate(y, High)][Saturate(x, D[0].High)];
  end;
  
  function TAuGraph._readheight: integer;
  begin
    Result:=Length;
  end;

  procedure TAuGraph.Transform;
  var x, y: integer;
  begin
    For y:=0 to High do
      For x:=0 to D[0].High do
        D[y][x]:= (D[y][x] and $0000FF00) shr 8;
    MinBrightness:=0;
    MaxBrightness:=255;
  end;
  
  procedure TAuColorTexture.Transform;
  begin
    MinBrightness:=0;
    MaxBrightness:=255;

    //do nothing
  end;

  
  Procedure TAuGraph.Inverse;
  var x, y: integer;
  begin
    For y:=0 to Height - 1 do
      For x:=0 to Width -1 do begin
        if (D[y][x] >= 0) and (D[y][x] <= 255)
          then D[y][x]:=255 - D[y][x];
      end;
  end;

  Procedure TAuGraph.SaveAsBMP(FileName: string);
  begin
    _SaveAsBmp(FileName, Self);
  end;
  
  procedure TAuGraph.TurnWhite;
  var x, y: integer;
  begin
    For y:=0 to Height - 1 do For x:=0 to Width -1 do D[y][x]:= 255;
  end;

  Procedure TAuGraph._SaveAsBMP(FileName: string; Duh:T2DAOI);
     Type
      TBmpHeader = packed record
        id: Word;
        fileSize,
        reserved1,
        bodyoffs,
        biSize,
        biWidth,
        biHeight : cardinal;
        biPlanes, //must be 1
        biBitCnt: Word;//must be 8 or 24
        biCompr, //must be 0
        biISize,
        biXResl,
        biYResl,
        biClrUsed,
        biClrImp: cardinal;
        palette: packed array[0..255, 0..3] of byte;
      end;
      function AlignToDword(i: integer): integer;
      begin
        While (i and $00000003) <> 0 do inc(i);
        Result:=i;
      end;
   var
    i, j, w    : Integer;
    header  : TBmpHeader;
    f: TFileStream;
    b: byte;
  begin
   Try
    FillChar(Header, SizeOf(Header), 0);
    Header.id:=$4D42;
    Header.biSize:=40;
    Header.biPlanes:=1;
    Header.biBitCnt:=8;
    Header.biCompr:=0;
    Header.bodyoffs:=SizeOf(Header);
    Header.biWidth:=Duh[0].Length;
    Header.biHeight:=Duh.Length;
    Header.biClrUsed:=256;
    Header.biClrImp:=0;
    Header.biXResl:=96;
    Header.biYResl:=96;
    w:=AlignToDword(Header.biWidth);
    Header.filesize:=Sizeof(Header) + (w * Header.biHeight);
    Header.BiISize:=(w * Header.biHeight);
    For i:=0 to 255 do begin
      Header.palette[i, 0]:=i;
      Header.palette[i, 1]:=i;
      Header.palette[i, 2]:=i;
      Header.palette[i, 3]:=0;
    end;
    F:=TFileStream.Create(filename, fmCreate or fmOpenWrite or fmShareExclusive);
    F.Write(header, sizeof(header));
    for i :=Header.biHeight - 1 downto 0 do begin
      for j := 0 to Header.biWidth-1 do	begin
        b:=clamp(Duh[i][j], 0, 255);
        f.Write(b, 1);
      end;
      if Header.biWidth <> w then f.Write(b, w-Header.biWidth);
        //  32- .
    end;
    SafeFree(F);
    //WriteLn('Image "',filename,'" saved. Width=',Header.biwidth,', height=',header.biheight);
   Except
     YellAndDie('   BMP  "%s"',[filename]);
   End;
  end;
  
  Procedure TAuColorMangaScan.SaveAsBMP(FileName: string);
     Type
      TBmpHeader = packed record
        id: Word;
        fileSize,
        reserved1,
        bodyoffs,
        biSize,
        biWidth,
        biHeight : cardinal;
        biPlanes, //must be 1
        biBitCnt: Word;//must be 8 or 24
        biCompr, //must be 0
        biISize,
        biXResl,
        biYResl,
        biClrUsed,
        biClrImp: cardinal;
      end;
      function AlignToDword(i: integer): integer;
      begin
        While (i and $00000001) <> 0 do inc(i);
        Result:=i;
      end;
   var
    i, j, w    : Integer;
    header  : TBmpHeader;
    f: TFileStream;
    b: integer;
  begin
   Try
    FillChar(Header, SizeOf(Header), 0);
    Header.id:=$4D42;
    Header.biSize:=Sizeof(Header);
    Header.biPlanes:=1;
    Header.biBitCnt:=24;
    Header.biCompr:=0;
    Header.bodyoffs:=SizeOf(Header);
    Header.biWidth:=Width;
    Header.biHeight:=Height;
    Header.biClrUsed:=0;
    Header.biClrImp:=0;
    Header.biXResl:=96;
    Header.biYResl:=96;
    w:=AlignToDword(Header.biWidth * 3);
    Header.filesize:=Sizeof(Header) + (w * Header.biHeight);
    Header.BiISize:=(w * Header.biHeight);
    F:=TFileStream.Create(filename, fmCreate or fmOpenWrite or fmShareExclusive);
    F.Write(header, sizeof(header));
    for i :=Header.biHeight - 1 downto 0 do begin
      for j := 0 to Header.biWidth-1 do	begin
        b:=Clamp(D[i][j], 0, 255)
          + Clamp(GV[i][j], 0, 255) shl 8 + Clamp(RV[i][j], 0, 255) shl 16;
        f.Write(b, 3);
      end;
      if (Header.biWidth *3) <> w then f.Write(b, w - (Header.biWidth*3));
        //  32- .
    end;
    SafeFree(F);
    //WriteLn('Image "',filename,'" saved. Width=',Header.biwidth,', height=',header.biheight);
   Except
     YellAndDie('   BMP  "%s"', [filename]);
   End;
  end;

  Function YellAndDie(s: string); OVERLOAD;
  begin
    raise Exception.Create(s);
  end;
  
  Function YellAndDie(s: string; c: array of const); OVERLOAD;
  begin
    raise Exception.Create(format(s, c) );
  end;



end.
