{
    This file is part of the SrtRetimer utility,
    Copyright (c) 2005 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the SRT and SSA handling class.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}


unit Script;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;
  
type
  TSrtScript = class (TStringList)
  protected
    FFileName: string;
    Loaded: boolean;
  public
    Modified: boolean;
    Constructor CreateFromFile(FileName: String);
    Constructor CreateNewFile(FileName: String);
    destructor Destroy; override;
    procedure Flush;
  end;
  
  TScript = class (TSrtScript)
  end;

  Function MsToSrtTime(ms: integer): string;
  Function SrtTimeToMs(s: string): integer;

var
  SRTErrormessage: string = '';

implementation
  {$iochecks on}

  Constructor TSrtScript.CreateNewFile(FileName: String);
  var
    srt: textFile;
  begin
    inherited Create;
    Try
      AssignFile(srt, FileName);
      ReWrite(srt);
    Except
      Raise Exception.Create(
          Format('%s'#10#13'   : %s',
            [FileName, (ExceptObject as Exception).Message]));
    End;
    CloseFile(srt);
    FFileName:=FileName;
    Loaded:=True;
    Modified:=True;
  end;

  Constructor TSrtScript.CreateFromFile(FileName: String);
  var
    srt: textFile;
    line, phrase: string;
    lnum, n: integer;
    Hr1, Min1, S1, Ms1, Hr2, Min2, S2, Ms2: integer;
  begin
    inherited Create;
   //Try
    Try
      AssignFile(srt, FileName);
      Reset(srt);
    Except
      Raise Exception.Create(
          Format('%s'#10#13'   : %s',
            [FileName, (ExceptObject as Exception).Message]));
    End;
    lnum:=0;
    while not eof(srt) do begin
      repeat
        ReadLn(srt, line);
        inc(lnum);
        line:=Trim(line);
      until eof(srt) or (line <> '');
      if eof(srt) then Break;
      Try
        n:=StrToInt(line);
      Except
        Raise Exception.Create(
          Format('%s'#10#13' %d:   ,  "%s"',
                 [FileName, lnum, line]));
      End;
      inc(lnum);
      if eof(srt) then Raise Exception.Create(
          Format('%s'#10#13' %d:   ,   ',
                                                             [FileName, lnum]));
      ReadLn(srt, line);
      Try
        //decoding the time code
        {example
00:20:46,783 --> 00:20:50,514
        }
        Hr1:=StrToInt(Copy(line, 1, 2));
        Hr2:=StrToInt(Copy(line, 18, 2));
        Min1:=StrToInt(Copy(line, 4, 2));
        Min2:=StrToInt(Copy(line, 21, 2));
        S1:=StrToInt(Copy(line, 7, 2));
        S2:=StrToInt(Copy(line, 24, 2));
        Ms1:=StrToInt(Copy(line, 10, 3));
        Ms2:=StrToInt(Copy(line, 27, 3));
        Ms1:=Ms1 + 1000 * (S1 + 60 * (Min1 + 60 * Hr1));
        Ms2:=Ms2 + 1000 * (S2 + 60 * (Min2 + 60 * Hr2)) - Ms1;
      Except
        if eof(srt) then Raise Exception.Create(
          Format('%s'#10#13' %d:   !',
                                            [FileName, lnum]));
      End;
      if eof(srt) then Raise Exception.Create(
          Format('%s'#10#13' %d:   ,   ',
                                                             [FileName, lnum]));
      Phrase:='';
      Repeat
        ReadLn(srt, line);
        inc(lnum);
        line:=Trim(line);
       if line = '' then Break;
        if Phrase<>'' then Phrase:=Phrase + #13#10;
        Phrase:=Phrase + line;
      Until eof(srt);
      Add(Format('%.10d%.10d%s',[Ms1, Ms2, Phrase]));
    end;
    CloseFile(srt);
    FFileName:=FileName;
    Loaded:=True;
   //Except
     //SRTErrormessage:=(ExceptObject as Exception).Message;
  // End;
  end;

  destructor TSrtScript.Destroy;
  begin
    if Loaded then Flush;
    inherited Destroy;
  end;

  procedure TSrtScript.Flush;
  var
    srt: textFile;
    line: string;
    n, ms1, ms2: integer;
  begin
    if not Modified then Exit;
    AssignFile(srt, FFilename);
    Rewrite(srt);
    For n:=0 to Count - 1 do begin
      WriteLn(srt, n + 1);
      line:=Self.Strings[n];
      ms1:=StrToInt(Copy(line, 1, 10));
      ms2:=ms1 + StrToInt(Copy(line, 11, 10));
      WriteLn(Srt, MsToSrtTime(ms1), ' --> ', MsToSrtTime(ms2));
      WriteLn(Srt, Copy(line, 21, Length(line) - 20));
      if n < (Count - 1) then WriteLn(srt);
    end;
    Close(srt);
  end;

  Function MsToSrtTime(ms: integer): string;
  begin
    Result:= Format('%.2d:%.2d:%.2d,%.3d',
    [(ms div 3600000) mod 60,
     (ms div 60000) mod 60,
     (ms div 1000) mod 60,
      ms mod 1000]);
  end;

  Function SrtTimeToMs(s: string): integer;
  var
    H, M, Se, Ms: integer;
  begin
    H:=StrToInt(Copy(s, 1, 2));
    M:=StrToInt(Copy(s, 4, 2));
    Se:=StrToInt(Copy(s, 7, 2));
    Ms:=StrToInt(Copy(s, 10, 3));
    Result:=Ms + 1000 * (Se + 60 * (M + 60 * H));
  end;

end.

