{
    This file is part of the SrtRetimer utility,
    Copyright (c) 2005 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the SRT and SSA handling class.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}


unit Script;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;
  
type

  { TSrtScript }

  TSrtScript = class (TStringList)
  protected
    FFileName: string;
    Loaded: boolean;
  public
    Modified: boolean;
    SSAHeader: string;
    Constructor CreateFromFile(FileName: String);
    Constructor CreateNewFile(FileName: String);
    function GetText(n: integer): string;
    function GetRefLine(n: integer): string;
    destructor Destroy; override;
    procedure Flush;
  end;
  
  TScript = class (TSrtScript)
  end;

  Function MsToSrtTime(ms: integer): string;
  Function SrtTimeToMs(s: string): integer;
  
  Function MsToSsaTime(ms: integer): string;
  Function SsaTimeToMs(s: string): integer;

  function CleanSRTLine(s: string): string;
  Function ConvertSSALine(s: string; Tstart, Tend: integer): string;
  function GetSsaStartTime(s: string): integer;
  function GetSsaEndTime(s: string): integer;

var
  SRTErrormessage: string = '';

implementation
uses SrtRetimer_Main;
  {$iochecks on}

  Constructor TSrtScript.CreateNewFile(FileName: String);
  var
    srt: textFile;
  begin
    inherited Create;
    FFileName:=FileName;
    Loaded:=True;
    Modified:=True;
  end;

  function TSrtScript.GetText(n: integer): string;
  var
    i, commanum: integer;
  begin
    commanum:=0;
    Result:='';
    for i:=1 to Length(Self[n]) do begin
      if Self[n][i]=',' then inc(commanum);
      if commanum=9 then Exit(Copy(Self[n], i + 1, Length(Self[n]) - i));
    end;
  end;

  Constructor TSrtScript.CreateFromFile(FileName: String);
  var
    srt: textFile;
    line, phrase: string;
    lnum, n: integer;
    t1, t2, Hr1, Min1, S1, Ms1, Hr2, Min2, S2, Ms2: integer;
    fe: String;
    BodyReached: boolean;
    Errors: boolean = false;
  begin
    inherited Create;
   Try
    fe:=UpperCase(ExtractFileExt(FileName));
    if (fe <> '.SRT') and (fe <> '.SSA') and (fe <> '.ASS')
      then Raise Exception.Create(
        Format('  : %s'#10#13'  *.SRT  *.SSA',
            [FileName]));
    Try
      AssignFile(srt, FileName);
      Reset(srt);
    Except
      Raise Exception.Create(
          Format('%s'#10#13'   : %s',
            [FileName, (ExceptObject as Exception).Message]));
    End;
    if fe = '.SRT' then begin
      lnum:=0;
      while not eof(srt) do begin
        repeat
          ReadLn(srt, line);
          inc(lnum);
          line:=Trim(line);
        until eof(srt) or (line <> '');
        if eof(srt) then Break;
        Try
          n:=StrToInt(line);
        Except
          Raise Exception.Create(
            Format('%s'#10#13' %d:   ,  "%s"',
                   [FileName, lnum, line]));
        End;
        inc(lnum);
        if eof(srt) then Raise Exception.Create(
            Format('%s'#10#13' %d:   ,   ',
                                                               [FileName, lnum]));
        ReadLn(srt, line);
        Try
          //decoding the time code
          {example
  00:20:46,783 --> 00:20:50,514
          }
          Hr1:=StrToInt(Copy(line, 1, 2));
          Hr2:=StrToInt(Copy(line, 18, 2));
          Min1:=StrToInt(Copy(line, 4, 2));
          Min2:=StrToInt(Copy(line, 21, 2));
          S1:=StrToInt(Copy(line, 7, 2));
          S2:=StrToInt(Copy(line, 24, 2));
          Ms1:=StrToInt(Copy(line, 10, 3));
          Ms2:=StrToInt(Copy(line, 27, 3));
          Ms1:=Ms1 + 1000 * (S1 + 60 * (Min1 + 60 * Hr1));
          Ms2:=Ms2 + 1000 * (S2 + 60 * (Min2 + 60 * Hr2)) - Ms1;
        Except
          if eof(srt) then Raise Exception.Create(
            Format('%s'#10#13' %d:   !',
                                              [FileName, lnum]));
        End;
        if eof(srt) then Raise Exception.Create(
            Format('%s'#10#13' %d:   ,   ',
                                                               [FileName, lnum]));
        Phrase:='';
        Repeat
          ReadLn(srt, line);
          inc(lnum);
          line:=Trim(line);
         if line = '' then Break;
          if Phrase<>'' then Phrase:=Phrase + #13#10;
          Phrase:=Phrase + line;
        Until eof(srt);
        Add(Format('%.10d%.10d,,,,,,,,,%s',[Ms1, Ms2, Phrase]));
      end;
    end
    else begin
      //SSA
      BodyReached:=false;
      SsaHeader:='';
      lnum:=0;
      while not eof(srt) do begin
        repeat
          ReadLn(srt, line);
          inc(lnum);
          SsaHeader:=SSaHeader + line + #13#10;
          if eof(srt) then Raise Exception.Create(
            Format(' %s'#13#10'   [Events]', [FileName]));
        until eof(srt) or (UpperCase(line) = '[EVENTS]');
        readln(srt, line);
        inc(lnum);
        SsaHeader:=SSaHeader + line;//Format: Marked, Start, End, Style, Name, MarginL, MarginR, MarginV, Effect, Text
        repeat
          ReadLn(srt, line);
          inc(lnum);
          line:=trim(line);
          if line='' then continue;
          t1:=GetSSaStartTime(line);
          t2:=GetSSaEndTime(line);
          if (t2 <= 0) or (t1 < 0) or (t2 < t1) then Raise Exception.Create(
            Format(' %s'#13#10'     %d :'#13#10'%s'
            , [FileName, lnum, line]));
          Add(Format('%.10d%.10d%s',[t1, t2 - t1, line]));
        until eof(srt);
      end;
    end;
    CloseFile(srt);
    FFileName:=FileName;
    Loaded:=True;
    Except
      ErrorMessage('   !', (ExceptObject as Exception).Message);
      Self.Clear;
      Try CloseFile(srt);Except End;
    End;
  end;

  destructor TSrtScript.Destroy;
  begin
    if Loaded then Flush;
    inherited Destroy;
  end;

  procedure TSrtScript.Flush;
  var
    srt: textFile;
    line: string;
    n, ms1, ms2: integer;
  begin
    if not Modified then Exit;
    if SSAHeader = ''
      then FFileName:=ChangeFileExt(FFileName, '.srt')
      else FFileName:=ChangeFileExt(FFileName, '.ssa');
    AssignFile(srt, FFilename);
    Rewrite(srt);
    if SSAHeader = '' then begin
      //srt
      For n:=0 to Count - 1 do begin
        WriteLn(srt, n + 1);
        line:=Self.Strings[n];
        ms1:=StrToInt(Copy(line, 1, 10));
        ms2:=ms1 + StrToInt(Copy(line, 11, 10));
        WriteLn(Srt, MsToSrtTime(ms1), ' --> ', MsToSrtTime(ms2));
        WriteLn(Srt, CleanSrtLine(GetText(n)));//Copy(line, 21, Length(line) - 20));
        if n < (Count - 1) then WriteLn(srt);
      end;
    end
    else begin
      //ssa
      WriteLn(srt, SSAHeader);
      For n:=0 to Count - 1 do begin
        line:=Self.Strings[n];
        ms1:=StrToInt(Copy(line, 1, 10));
        ms2:=ms1 + StrToInt(Copy(line, 11, 10));
        WriteLn(Srt, ConvertSSALine(Copy(line, 21, Length(line) - 20), ms1, ms2));
      end;
    end;
    Close(srt);
  end;

  Function MsToSrtTime(ms: integer): string;
  begin
    Result:= Format('%.2d:%.2d:%.2d,%.3d',
    [(ms div 3600000) mod 60,
     (ms div 60000) mod 60,
     (ms div 1000) mod 60,
      ms mod 1000]);
  end;
  
  Function MsToSsaTime(ms: integer): string;
  begin
    Result:= Format('%.1d:%.2d:%.2d.%.2d',
    [(ms div 3600000) mod 60,
     (ms div 60000) mod 60,
     (ms div 1000) mod 60,
      (ms mod 1000) div 10]);
  end;

  Function SrtTimeToMs(s: string): integer;
  var
    H, M, Se, Ms: integer;
  begin
    H:=StrToInt(Copy(s, 1, 2));
    M:=StrToInt(Copy(s, 4, 2));
    Se:=StrToInt(Copy(s, 7, 2));
    Ms:=StrToInt(Copy(s, 10, 3));
    Result:=Ms + 1000 * (Se + 60 * (M + 60 * H));
  end;
  
//0:05:12.81
  Function SsaTimeToMs(s: string): integer;
  var
    H, M, Se, Ms: integer;
  begin
    //0:00:00.00
    H:=StrToInt(Copy(s, 1, 1));
    M:=StrToInt(Copy(s, 3, 2));
    Se:=StrToInt(Copy(s, 6, 2));
    Ms:=StrToInt(Copy(s, 9, 2));
    Result:=Ms * 10 + 1000 * (Se + 60 * (M + 60 * H));
  end;
  
  function GetSsaStartTime(s: string): integer;
  var
    i, commanum: integer;
  begin
    commanum:=0;
    Result:=-1;
    for i:=1 to Length(s) do begin
      if s[i]=',' then inc(commanum);
      if commanum=1 then
        if Copy(s, i + 11, 1) <> ','
          then Exit(-1)
          else Exit(SsaTimeToMs(Copy(s, i + 1, 10)));
    end;
  end;

  function GetSsaEndTime(s: string): integer;
  var
    i, commanum: integer;
  begin
    commanum:=0;
    Result:=-1;
    for i:=1 to Length(s) do begin
      if s[i]=',' then inc(commanum);
      if commanum=2 then begin
        if Copy(s, i + 11, 1) <> ','
          then Exit(-1)
          else Exit(SsaTimeToMs(Copy(s, i + 1, 10)));
      end;
    end;
  end;

  function ConvertSSALine(s: string; Tstart, Tend: integer): string;
  var
    p: array[0..9] of string;
    i, n: integer;
  begin
    For i:=0 to 9 do p[i]:='';
    i:=0;
    For n:=1 to Length(s) do begin
      if s[n]=',' then begin
        if i < 9 then inc(i)
                 else p[9]:=p[9] + ',';
      end
      else p[i]:=p[i] + s[n];
    end;
    p[1]:=MsToSsaTime(TStart);
    p[2]:=MsToSsaTime(TEnd);
    if p[0]='' then p[0]:='Dialogue: Marked=0';
    if p[3]='' then p[3]:='Default';
    For i:=5 to 7 do
      if Length(p[i]) <> 4 then p[i]:='0000';
    p[9]:=StringReplace(p[9], #13 ,'\N', [rfReplaceAll]);
    p[9]:=StringReplace(p[9], #10 ,'', [rfReplaceAll]);
    Result:='';
    For i:=0 to 8 do Result:=result + p[i]+',';
    Result:=Result + p[9];
  end;

  function CleanSRTLine(s: string): string;
  begin
    Result:=StringReplace(s, '\n', #13#10, [rfReplaceAll]);
    Result:=StringReplace(Result, '\N', #13#10, [rfReplaceAll]);
    while Pos(Result, #13#10#13#10) > 0 do
      Result:=StringReplace(Result, '\N', #13#10, [rfReplaceAll]);
    Result:=StringReplace(Result, '<b>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '<B>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '</B>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '</b>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '<i>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '<I>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '</I>', '', [rfReplaceAll]);
    Result:=StringReplace(Result, '</i>', '', [rfReplaceAll]);
  end;
  
  Function TotalCleanLine(s: string): string;
  begin
    Result:=CleanSRTLine(s);
    Result:=StringReplace(Result, #13, ' ', [rfReplaceAll]);
    Result:=StringReplace(Result, #10, '', [rfReplaceAll]);
  end;

  function TSrtScript.GetRefLine(n: integer): string;
  var l: integer;
  begin
    l:=StrToInt(Copy(Strings[n],11,10));
    Result:=format('%s (%.1d.%.3d)  %s',[
      MsToSrtTime(StrToInt(Copy(Strings[n],1,10))),
      l div 1000, l mod 1000,
      TotalCleanLine(GetText(n))
     ]);
  end;

end.

