//------------------------------------------------------------------------
// Pixel shader demonstrator compiled by Cheb
// from Jan Horn's Delphi OpenGL template
// and the CyberDemon's C++ example program from GameDev.ru
// (Note: you cannot use program examples from http://gamedev.ru/
//  unless you can read in Russian)

// Controls:
// Space switches shiny mode,
// Enter switches lighting mode,
// "A": slower, "S" faster.
//
// Author      : Cheb, compiled from Jan Horn's and CyberDemon's works
// Email       : cheb@internets.ru
// Website     : http://www.sulaco.co.za  (site on which this example is published)
//               http://chebmaster.narod.ru/  (Cheb's site)
// Date        : 06 September 2002
// Version     : n/a
// Description : Examples hardware bump mapping on GeForce1 or any other
//               video card, supporting NV_REGISTER_COMBINERS extension
// (note: NVREGCOMB.PAS included, because the standard OpenGL12.pas does not
//    contain necessary constants and procedures !      )

// Adobe Photoshop plugin for making bump textures from height maps
// is available on nVidia site http://developer.nvidia.com/ - dig deeper,
// and you will find it. :)

//P.S. There may be much more complex effects, but my GeForce2 has only 2
// texture units, and supports only 2 combiners at time - so any more
// complex effect would take two or more passes, and I didn't bothered
// implementing them.
//If you have GeForce 3 or 4, you may try to experiment - I think, these
// video cards should have more texture units and combiners...
//------------------------------------------------------------------------
program regcombiners_bumpmap;

uses
  Windows,
  Messages,
  OpenGL12,
  TGATexture,
  NVREGCOMB;

const
  WND_TITLE = 'Bump Mapping demonstrator';
  FPS_TIMER = 1;                     // Timer to calculate FPS
  FPS_INTERVAL = 500;                // Calculate FPS every 500 ms

var
  h_Wnd  : HWND;                     // Global window handle
  h_DC   : HDC;                      // Global device context
  h_RC   : HGLRC;                    // OpenGL rendering context
  keys : Array[0..255] of Boolean;   // Holds keystrokes
  FPSCount : Integer = 0;            // Counter for FPS
  ElapsedTime : Integer;             // Elapsed time between frames

  // Textures
  idBase,
  idBump : glUint;
  MaxCombiners: glInt;
  ShinyMode: integer = 0;
  Speed: integer = 1;
  LightMode: integer = 0;


{$R *.RES}

//procedure glBindTexture(target: GLenum; texture: GLuint); stdcall; external opengl32;

{------------------------------------------------------------------}
{  Function to convert int to string. (No sysutils = smaller EXE)  }
{------------------------------------------------------------------}
function IntToStr(Num : Integer) : String;  // using SysUtils increase file size by 100K
begin
  Str(Num, result);
end;


{------------------------------------------------------------------}
{  Function to draw the actual scene                               }
{------------------------------------------------------------------}
procedure glDraw();
var I, C : Integer;
x,y,z: single;
begin
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);    // Clear The Screen And The Depth Buffer
  glLoadIdentity();   // Reset The View
  glShadeModel(GL_SMOOTH);
  glRotatef(ElapsedTime/100, 0, 0, 1);
  glTranslatef(0, 0, -ElapsedTime/15000);
  glDepthFunc(GL_LESS);

  glActiveTextureARB(GL_TEXTURE1_ARB);
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, idBase);

  glActiveTextureARB(GL_TEXTURE0_ARB);
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, idBump);

  glEnable(GL_REGISTER_COMBINERS_NV);


 Case ShinyMode of
  0: begin
    //Specular bump mapping with only 1 combiner
    glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 1);

    //RUS:   General Combiner -      
    //    Dot3, ,   A  Bump-,
    // ,       [-1,1]:

    //ENG: Now we will program the first General Combiner - we need to multiply
    //light vector by the first texture using Dot3 operation, so the A variable
    //vill point to the Bump texture
    //(note, that its range will be expanded to [-1; 1]):

    glCombinerInputNV(
      GL_COMBINER0_NV,    // num combiner
      GL_RGB,             // portion
      GL_VARIABLE_A_NV,   // variable
      GL_TEXTURE0_ARB,    // input
      GL_EXPAND_NORMAL_NV,// mapping
      GL_RGB              // component usage
    );

    //RUS:  B    :
    //ENG: We will set the B variable to light vector:

    glCombinerInputNV(
      GL_COMBINER0_NV,
      GL_RGB,
      GL_VARIABLE_B_NV,
      GL_PRIMARY_COLOR_NV,
      GL_EXPAND_NORMAL_NV,
      GL_RGB
    );
    glCombinerOutputNV(
      GL_COMBINER0_NV,
      GL_RGB,
      GL_SPARE0_NV,       // AB output
      GL_DISCARD_NV,      // CD output
      GL_DISCARD_NV,      // sum output
      GL_NONE,            // scale
      GL_NONE,            // bias
      GL_TRUE,            // ab dot product
      GL_FALSE,           // cd dot product
      GL_FALSE            // muxsum
    );

    //RUS:  Final Combiner -   A  B,    
    // Base . A*B     SPARE0 ,  Final Combiner
    //  A  SPARE0,   B  Base ,
    //   (.  ) res = A*B + (1-A)*C + D = A*B;

    //ENG: Now we need to setup the final combiner  - we need to multiply A and B
    //and the multiply result by the Base texture. A*B now is placed in SPARE0,
    //so for FinalCombiner we must point A variable to SPARE0 ant B variable to
    //Base texture, as result we will get res = A*B + (1-A)*C + D = A*B

    glFinalCombinerInputNV(GL_VARIABLE_A_NV,
        GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    glFinalCombinerInputNV(GL_VARIABLE_B_NV,
        GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    glFinalCombinerInputNV(GL_VARIABLE_C_NV,
        GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    glFinalCombinerInputNV(GL_VARIABLE_D_NV,
          GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    glFinalCombinerInputNV(GL_VARIABLE_E_NV,
        GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    glFinalCombinerInputNV(GL_VARIABLE_F_NV,
        GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
  end;
  1..3: begin
    //Shiny effects with 2 combiners
    glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 2);
    glCombinerInputNV(
      GL_COMBINER0_NV,    // num combiner
      GL_RGB,             // portion
      GL_VARIABLE_A_NV,   // variable
      GL_TEXTURE0_ARB,    // input
      GL_EXPAND_NORMAL_NV,// mapping
      GL_RGB              // component usage
    );
    glCombinerInputNV(
      GL_COMBINER0_NV,
      GL_RGB,
      GL_VARIABLE_B_NV,
      GL_PRIMARY_COLOR_NV,
      GL_EXPAND_NORMAL_NV,
      GL_RGB
    );
    glCombinerOutputNV(
      GL_COMBINER0_NV,
      GL_RGB,
      GL_SPARE0_NV,       // AB output
      GL_DISCARD_NV,      // CD output
      GL_DISCARD_NV,      // sum output
      GL_NONE,            // scale
      GL_NONE,            // bias
      GL_TRUE,            // ab dot product
      GL_FALSE,           // cd dot product
      GL_FALSE            // muxsum
    );

    //Here, in Combiner 1, we will do something weird.
    //We will Dot3 the result in SPARE0 with itself
    //and place it to SPARE1.
    //Mathematically, this is a nonsemse, but it gives a nice shiny highlight...
    glCombinerInputNV(
      GL_COMBINER1_NV,    // num combiner
      GL_RGB,             // portion
      GL_VARIABLE_A_NV,   // variable
      GL_SPARE0_NV,    // input
      GL_UNSIGNED_IDENTITY_NV,//GL_EXPAND_NORMAL_NV,// mapping
      GL_RGB              // component usage
    );
    glCombinerInputNV(
      GL_COMBINER1_NV,
      GL_RGB,
      GL_VARIABLE_B_NV,
      GL_SPARE0_NV,
      GL_UNSIGNED_IDENTITY_NV,//GL_EXPAND_NORMAL_NV,
      GL_RGB
    );


    Case ShinyMode of
      //in Mode1 we will just add SPARE1 to the result...

      1: begin
         glCombinerOutputNV(
          GL_COMBINER1_NV,
          GL_RGB,
          GL_SPARE1_NV,       // AB output
          GL_DISCARD_NV,      // CD output
          GL_DISCARD_NV,      // sum output
          GL_NONE,            // scale
          GL_NONE,            // bias
          GL_False,            // ab dot product
          GL_FALSE,           // cd dot product
          GL_FALSE            // muxsum
        );

        glFinalCombinerInputNV(GL_VARIABLE_A_NV,
            GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_B_NV,
            GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_C_NV,
            GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_D_NV,
              GL_SPARE1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
          //just add shiny highlight (from SPARE1) to the result

        glFinalCombinerInputNV(GL_VARIABLE_E_NV,
            GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_F_NV,
            GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
      end;
      2: begin
        //and in Mode2 we will add SPARE1*SPARE1 to the result.
        //Because range of that value is [0..1], so by taking sqare of it we
        //making the same [0..1] but with more sharp highlight...
         glCombinerOutputNV(
          GL_COMBINER1_NV,
          GL_RGB,
          GL_SPARE1_NV,       // AB output
          GL_DISCARD_NV,      // CD output
          GL_DISCARD_NV,      // sum output
          GL_NONE,            // scale
          GL_NONE,            // bias
          GL_TRUE,            // ab dot product
          GL_FALSE,           // cd dot product
          GL_FALSE            // muxsum
        );
        glFinalCombinerInputNV(GL_VARIABLE_A_NV,
            GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_B_NV,
            GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_C_NV,
            GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_D_NV,
              GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
          //just add shiny highlight (from SPARE1) to the result

        glFinalCombinerInputNV(GL_VARIABLE_E_NV,
            GL_SPARE1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_F_NV,
            GL_SPARE1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
      end;
      3: begin
        //in mode 3 we will do same thing as in mode 2, except that
        //we will change output of General Combiner 1:
        //we will shift it toward negative values, thus making highlight
        //even more sharp... 
        glCombinerOutputNV(
          GL_COMBINER1_NV,
          GL_RGB,
          GL_SPARE1_NV,       // AB output
          GL_DISCARD_NV,      // CD output
          GL_DISCARD_NV,      // sum output
          GL_NONE,//SCALE_BY_TWO_NV,            // scale
          GL_BIAS_BY_NEGATIVE_ONE_HALF_NV, // bias
          GL_TRUE,            // ab dot product
          GL_FALSE,           // cd dot product
          GL_FALSE            // muxsum
        );
        glFinalCombinerInputNV(GL_VARIABLE_A_NV,
            GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_B_NV,
            GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_C_NV,
            GL_ZERO, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_D_NV,
              GL_E_TIMES_F_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
          //just add shiny highlight (from SPARE1) to the result

        glFinalCombinerInputNV(GL_VARIABLE_E_NV,
            GL_SPARE1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);

        glFinalCombinerInputNV(GL_VARIABLE_F_NV,
            GL_SPARE1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
      end;
    End;
  end;
 End;


   glBegin(GL_QUADS);
   //RUS:    (  PRIMARY_COLOR),   
   //  ()   :

   //ENG: The light vector will be encoded in the PRIMARY_COLOR.
   //In real life it will be calculated by some way, but here I just rotate
   //light vectors, uniformly in light mode 0 and independently in mode 1.
   //on different speed for each of four vertexes.
   //I don't believe this gives correct result (interpolated color values
   //vill be NOT normalized vectors) but in the real life, difference
   //between light normal vectors is small enough, and this error is unnoticeable.

  if LightMode = 1 then begin
   x:=(1+ cos(ElapsedTime/200)*sin(ElapsedTime/1200))/2;
   y:=(1+ sin(ElapsedTime/200)*sin(ElapsedTime/1200))/2;
   z:=(1+ abs(cos(ElapsedTime/1200)))/2;
  end
  else begin
   x:=(1 + 0.954 * -cos(((Pi/180)*ElapsedTime)/100))/2;
   y:=(1 + 0.954 * -sin(((Pi/180)*ElapsedTime)/100))/2;
   z:=(1 + 0.3)/2;
  end;
   glColor3f(x, y, z);
   glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 0,0);
   glMultiTexCoord2fARB(GL_TEXTURE1_ARB, 0,0);
   glVertex3f(-2.0, -2.0, 0);

  if LightMode = 1 then begin
   x:=(1+ cos(ElapsedTime/250)*sin(ElapsedTime/1666))/2;
   y:=(1+ sin(ElapsedTime/250)*sin(ElapsedTime/1666))/2;
   z:=(1+ abs(cos(ElapsedTime/1666)))/2;
   glColor3f(x, y, z);
  end;
   glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 3,0);
   glMultiTexCoord2fARB(GL_TEXTURE1_ARB, 1,0);
   glVertex3f(-2.0,  2.0, 0);

  if LightMode = 1 then begin
   x:=(1+ cos(ElapsedTime/300)*sin(ElapsedTime/2000))/2;
   y:=(1+ sin(ElapsedTime/300)*sin(ElapsedTime/2000))/2;
   z:=(1+ abs(cos(ElapsedTime/2000)))/2;
   glColor3f(x, y, z);
  end;
   glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 3,3);
   glMultiTexCoord2fARB(GL_TEXTURE1_ARB, 1,1);
   glVertex3f( 2.0,  2.0, 0);

  if LightMode = 1 then begin
   x:=(1+ cos(ElapsedTime/466)*sin(ElapsedTime/1000))/2;
   y:=(1+ sin(ElapsedTime/466)*sin(ElapsedTime/1000))/2;
   z:=(1+ abs(cos(ElapsedTime/1000)))/2;
   glColor3f(x, y, z);
  end; 
   glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 0,3);
   glMultiTexCoord2fARB(GL_TEXTURE1_ARB, 0,1);
   glVertex3f( 2.0, -2.0, 0);
   glEnd();

   glDisable(GL_REGISTER_COMBINERS_NV);

   glDisable(GL_TEXTURE_2D);
   glActiveTextureARB(GL_TEXTURE0_ARB);
   glDisable(GL_TEXTURE_2D);
end;


{------------------------------------------------------------------}
{  Initialise OpenGL                                               }
{------------------------------------------------------------------}
procedure glInit();

begin
  glClearColor(0, 0, 0, 0.0); 	   // Black Background
  glShadeModel(GL_FLAT);
  glClearDepth(1.0);                       // Depth Buffer Setup
  glEnable(GL_DEPTH_TEST);                 // Enable Depth Buffer
  glDepthFunc(GL_LESS);		           // The Type Of Depth Test To Do

  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);   //Realy Nice perspective calculations

  LoadTexture('base.tga', idbase);
  LoadTexture('bump.tga', idbump);
  glGetIntegerv( GL_MAX_GENERAL_COMBINERS_NV, @MaxCombiners);
end;


{------------------------------------------------------------------}
{  Handle window resize                                            }
{------------------------------------------------------------------}
procedure glResizeWnd(Width, Height : Integer);
begin
  if (Height = 0) then                // prevent divide by zero exception
    Height := 1;
  glViewport(0, 0, Width, Height);    // Set the viewport for the OpenGL window
  glMatrixMode(GL_PROJECTION);        // Change Matrix Mode to Projection
  glLoadIdentity();                   // Reset View
  gluPerspective(45.0, Width/Height, 0.1, 500.0);  // Do the perspective calculations. Last value = max clipping depth

  glMatrixMode(GL_MODELVIEW);         // Return to the modelview matrix
  glLoadIdentity();                   // Reset View
end;

procedure glKillWnd(Fullscreen : Boolean); FORWARD;
{------------------------------------------------------------------}
{  Processes all the keystrokes                                    }
{------------------------------------------------------------------}
procedure ProcessKeys;
begin
  if keys[Ord(' ')] then Inc(ShinyMode);
  If ShinyMode > 3 then ShinyMode:=0;
  if (ShinyMode > 0) and (MaxCombiners < 2) then begin
    glKillWnd(False);
    MessageBox(0, 'Sorry, but your video card suppors only one combiner at once. Shiny effects need two. (like on GeForce2)', 'Error', MB_OK or MB_ICONERROR);
    Exit;
  end;
  keys[Ord(' ')] :=FALSE;
  if keys[Ord('A')] then begin
    Dec(Speed);
    keys[Ord('A')] :=FALSE;
  end;
  if keys[Ord('S')] then begin
    Inc(Speed);
    keys[Ord('S')] :=FALSE;
  end;
  if keys[Ord(#13)] then begin
    LightMode:=LightMode + 1;
    if LightMode > 1 then LightMode:=0;
    keys[Ord(#13)] :=FALSE;
  end;
end;


{------------------------------------------------------------------}
{  Determines the applications response to the messages received  }
{------------------------------------------------------------------}
function WndProc(hWnd: HWND; Msg: UINT;  wParam: WPARAM;  lParam: LPARAM): LRESULT; stdcall;
begin
  case (Msg) of
    WM_CREATE:
      begin
        // Insert stuff you want executed when the program starts
      end;
    WM_CLOSE:
      begin
        PostQuitMessage(0);
        Result := 0
      end;
    WM_KEYDOWN:       // Set the pressed key (wparam) to equal true so we can check if its pressed
      begin
        keys[wParam] := True;
        Result := 0;
      end;
    WM_KEYUP:         // Set the released key (wparam) to equal false so we can check if its pressed
      begin
        keys[wParam] := False;
        Result := 0;
      end;
    WM_SIZE:          // Resize the window with the new width and height
      begin
        glResizeWnd(LOWORD(lParam),HIWORD(lParam));
        Result := 0;
      end;
    WM_TIMER :                     // Add code here for all timers to be used.
      begin
        if wParam = FPS_TIMER then
        begin
          FPSCount :=Round(FPSCount * 1000/FPS_INTERVAL);   // calculate to get per Second incase intercal is less or greater than 1 second
          If ShinyMode = 0 then
            SetWindowText(h_Wnd,PChar('[FPS:' + intToStr(FPSCount) + ']   ' + WND_TITLE))
          else
            SetWindowText(h_Wnd,PChar('[FPS:' + intToStr(FPSCount) + ']   '
            + WND_TITLE + ' - Shiny mode #' + IntToStr(ShinyMode)));
          FPSCount := 0;
          Result := 0;
        end;
      end;
    else
      Result := DefWindowProc(hWnd, Msg, wParam, lParam);    // Default result if nothing happens
  end;
end;


{---------------------------------------------------------------------}
{  Properly destroys the window created at startup (no memory leaks)  }
{---------------------------------------------------------------------}
procedure glKillWnd(Fullscreen : Boolean);
begin
  if Fullscreen then             // Change back to non fullscreen
  begin
    ChangeDisplaySettings(devmode(nil^), 0);
    ShowCursor(True);
  end;

  // Makes current rendering context not current, and releases the device
  // context that is used by the rendering context.
  if (not wglMakeCurrent(h_DC, 0)) then
    MessageBox(0, 'Release of DC and RC failed!', 'Error', MB_OK or MB_ICONERROR);

  // Attempts to delete the rendering context
  if (not wglDeleteContext(h_RC)) then
  begin
    MessageBox(0, 'Release of rendering context failed!', 'Error', MB_OK or MB_ICONERROR);
    h_RC := 0;
  end;

  // Attemps to release the device context
  if ((h_DC > 0) and (ReleaseDC(h_Wnd, h_DC) = 0)) then
  begin
    MessageBox(0, 'Release of device context failed!', 'Error', MB_OK or MB_ICONERROR);
    h_DC := 0;
  end;

  // Attempts to destroy the window
  if ((h_Wnd <> 0) and (not DestroyWindow(h_Wnd))) then
  begin
    MessageBox(0, 'Unable to destroy window!', 'Error', MB_OK or MB_ICONERROR);
    h_Wnd := 0;
  end;

  // Attempts to unregister the window class
  if (not UnRegisterClass('OpenGL', hInstance)) then
  begin
    MessageBox(0, 'Unable to unregister window class!', 'Error', MB_OK or MB_ICONERROR);
    hInstance := 0;
  end;
end;


{--------------------------------------------------------------------}
{  Creates the window and attaches a OpenGL rendering context to it  }
{--------------------------------------------------------------------}
function glCreateWnd(Width, Height : Integer; Fullscreen : Boolean; PixelDepth : Integer) : Boolean;
var
  wndClass : TWndClass;         // Window class
  dwStyle : DWORD;              // Window styles
  dwExStyle : DWORD;            // Extended window styles
  dmScreenSettings : DEVMODE;   // Screen settings (fullscreen, etc...)
  PixelFormat : GLuint;         // Settings for the OpenGL rendering
  h_Instance : HINST;           // Current instance
  pfd : TPIXELFORMATDESCRIPTOR;  // Settings for the OpenGL window
begin
  h_Instance := GetModuleHandle(nil);       //Grab An Instance For Our Window
  ZeroMemory(@wndClass, SizeOf(wndClass));  // Clear the window class structure

  with wndClass do                    // Set up the window class
  begin
    style         := CS_HREDRAW or    // Redraws entire window if length changes
                     CS_VREDRAW or    // Redraws entire window if height changes
                     CS_OWNDC;        // Unique device context for the window
    lpfnWndProc   := @WndProc;        // Set the window procedure to our func WndProc
    hInstance     := h_Instance;
    hCursor       := LoadCursor(0, IDC_ARROW);
    lpszClassName := 'OpenGL';
  end;

  if (RegisterClass(wndClass) = 0) then  // Attemp to register the window class
  begin
    MessageBox(0, 'Failed to register the window class!', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit
  end;

  // Change to fullscreen if so desired
  if Fullscreen then
  begin
    ZeroMemory(@dmScreenSettings, SizeOf(dmScreenSettings));
    with dmScreenSettings do begin              // Set parameters for the screen setting
      dmSize       := SizeOf(dmScreenSettings);
      dmPelsWidth  := Width;                    // Window width
      dmPelsHeight := Height;                   // Window height
      dmBitsPerPel := PixelDepth;               // Window color depth
      dmFields     := DM_PELSWIDTH or DM_PELSHEIGHT or DM_BITSPERPEL;
    end;

    // Try to change screen mode to fullscreen
    if (ChangeDisplaySettings(dmScreenSettings, CDS_FULLSCREEN) = DISP_CHANGE_FAILED) then
    begin
      MessageBox(0, 'Unable to switch to fullscreen!', 'Error', MB_OK or MB_ICONERROR);
      Fullscreen := False;
    end;
  end;

  // If we are still in fullscreen then
  if (Fullscreen) then
  begin
    dwStyle := WS_POPUP or                // Creates a popup window
               WS_CLIPCHILDREN            // Doesn't draw within child windows
               or WS_CLIPSIBLINGS;        // Doesn't draw within sibling windows
    dwExStyle := WS_EX_APPWINDOW;         // Top level window
    ShowCursor(False);                    // Turn of the cursor (gets in the way)
  end
  else
  begin
    dwStyle := WS_OVERLAPPEDWINDOW or     // Creates an overlapping window
               WS_CLIPCHILDREN or         // Doesn't draw within child windows
               WS_CLIPSIBLINGS;           // Doesn't draw within sibling windows
    dwExStyle := WS_EX_APPWINDOW or       // Top level window
                 WS_EX_WINDOWEDGE;        // Border with a raised edge
  end;

  // Attempt to create the actual window
  h_Wnd := CreateWindowEx(dwExStyle,      // Extended window styles
                          'OpenGL',       // Class name
                          WND_TITLE,      // Window title (caption)
                          dwStyle,        // Window styles
                          0, 0,           // Window position
                          Width, Height,  // Size of window
                          0,              // No parent window
                          0,              // No menu
                          h_Instance,     // Instance
                          nil);           // Pass nothing to WM_CREATE
  if h_Wnd = 0 then
  begin
    glKillWnd(Fullscreen);                // Undo all the settings we've changed
    MessageBox(0, 'Unable to create window!', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Try to get a device context
  h_DC := GetDC(h_Wnd);
  if (h_DC = 0) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to get a device context!', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Settings for the OpenGL window
  with pfd do
  begin
    nSize           := SizeOf(TPIXELFORMATDESCRIPTOR); // Size Of This Pixel Format Descriptor
    nVersion        := 1;                    // The version of this data structure
    dwFlags         := PFD_DRAW_TO_WINDOW    // Buffer supports drawing to window
                       or PFD_SUPPORT_OPENGL // Buffer supports OpenGL drawing
                       or PFD_DOUBLEBUFFER;  // Supports double buffering
    iPixelType      := PFD_TYPE_RGBA;        // RGBA color format
    cColorBits      := PixelDepth;           // OpenGL color depth
    cRedBits        := 0;                    // Number of red bitplanes
    cRedShift       := 0;                    // Shift count for red bitplanes
    cGreenBits      := 0;                    // Number of green bitplanes
    cGreenShift     := 0;                    // Shift count for green bitplanes
    cBlueBits       := 0;                    // Number of blue bitplanes
    cBlueShift      := 0;                    // Shift count for blue bitplanes
    cAlphaBits      := 0;                    // Not supported
    cAlphaShift     := 0;                    // Not supported
    cAccumBits      := 0;                    // No accumulation buffer
    cAccumRedBits   := 0;                    // Number of red bits in a-buffer
    cAccumGreenBits := 0;                    // Number of green bits in a-buffer
    cAccumBlueBits  := 0;                    // Number of blue bits in a-buffer
    cAccumAlphaBits := 0;                    // Number of alpha bits in a-buffer
    cDepthBits      := 16;                   // Specifies the depth of the depth buffer
    cStencilBits    := 0;                    // Turn off stencil buffer
    cAuxBuffers     := 0;                    // Not supported
    iLayerType      := PFD_MAIN_PLANE;       // Ignored
    bReserved       := 0;                    // Number of overlay and underlay planes
    dwLayerMask     := 0;                    // Ignored
    dwVisibleMask   := 0;                    // Transparent color of underlay plane
    dwDamageMask    := 0;                     // Ignored
  end;

  // Attempts to find the pixel format supported by a device context that is the best match to a given pixel format specification.
  PixelFormat := ChoosePixelFormat(h_DC, @pfd);
  if (PixelFormat = 0) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to find a suitable pixel format', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Sets the specified device context's pixel format to the format specified by the PixelFormat.
  if (not SetPixelFormat(h_DC, PixelFormat, @pfd)) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to set the pixel format', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Create a OpenGL rendering context
  h_RC := wglCreateContext(h_DC);
  if (h_RC = 0) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to create an OpenGL rendering context', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;



  // Makes the specified OpenGL rendering context the calling thread's current rendering context
  if (not wglMakeCurrent(h_DC, h_RC)) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to activate OpenGL rendering context', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  //Determining the procedure addresses for OpenGL extensions
  //(should be done after setting current rendering context)
  ReadExtensions;
  ReadNVREGCOMBExtension;

  if not (Assigned(@glCombinerParameteriNV) and Assigned(@glActiveTextureARB))
  //If there is no required extensions, procedure addresses will be NIL.
  //This is not method you really should use (there is a  other, "legal" way
  // to check accessibility of extensions), but , as I said before, I'm lazy.
  then begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Sorry, but your video card or OpenGL driver does not support the nVidia Register Combiners extension', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;


  // Initializes the timer used to calculate the FPS
  SetTimer(h_Wnd, FPS_TIMER, FPS_INTERVAL, nil);

  // Settings to ensure that the window is the topmost window
  ShowWindow(h_Wnd, SW_SHOW);
  SetForegroundWindow(h_Wnd);
  SetFocus(h_Wnd);

  // Ensure the OpenGL window is resized properly
  glResizeWnd(Width, Height);
  glInit();

  Result := True;
end;


{--------------------------------------------------------------------}
{  Main message loop for the application                             }
{--------------------------------------------------------------------}
function WinMain(hInstance : HINST; hPrevInstance : HINST;
                 lpCmdLine : PChar; nCmdShow : Integer) : Integer; stdcall;
var
  msg : TMsg;
  finished : Boolean;
  CurrentTick, LastTick: DWord;
  I: GLInt;
begin
  finished := False;

  // Perform application initialization:
  if not glCreateWnd(800, 600, FALSE, 32) then
  begin
    Result := 0;
    Exit;
  end;

  LastTick := GetTickCount();            // Get Time when demo started

  // Main message loop:
  while not finished do
  begin
    if (PeekMessage(msg, 0, 0, 0, PM_REMOVE)) then // Check if there is a message for this window
    begin
      if (msg.message = WM_QUIT) then     // If WM_QUIT message received then we are done
        finished := True
      else
      begin                               // Else translate and dispatch the message to this window
  	TranslateMessage(msg);
        DispatchMessage(msg);
      end;
    end
    else
    begin
      Inc(FPSCount);                      // Increment FPS Counter
      CurrentTick:=GetTickCount;
      ElapsedTime:=ElapsedTime + Speed * (CurrentTick - LastTick);
      LastTick:=CurrentTick;
      If ElapsedTime>60000 then begin
        LastTick:=GetTickCount;
        ElapsedTime:=0;
      end;
      If ElapsedTime<0 then begin
        LastTick:=GetTickCount;
        ElapsedTime:=60000;
      end;
      glDraw();                           // Draw the scene
      SwapBuffers(h_DC);                  // Display the scene

      if (keys[VK_ESCAPE]) then           // If user pressed ESC then set finised TRUE
        finished := True
      else
        ProcessKeys;                      // Check for any other key Pressed
    end;
  end;
  glKillWnd(FALSE);
  Result := msg.wParam;
end;


begin
  InitOpenGL;
  WinMain( hInstance, hPrevInst, CmdLine, CmdShow );
end.
