{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the texture manager classes.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}
 
{$include mo_globaldefs.h}

unit mo_textures;


interface

uses un_typedefs,  mo_hub, mo_classes, mo_pictures, mo_globopts, mo_indexer;

Type
  TAlphaKind = (No_Alpha, Threshold_Alpha, Gradient_Alpha);
  
  TMotherTexture = class (TTrulyPersistent)
    //base class for all textures.
  protected
    f_unclaimed: boolean;
    f_sourcename: AnsiString;
    f_thumb: T2DArrayOfDword;
    f_index: integer;
    f_Token, f_thumbToken: int64;
    f_name, f_thumbname: glUint;
    f_thumbformat: glInt;
    f_target: glEnum;
    f_priority: glFloat;
    f_ReloadUrgency: float;
    f_HasMips: boolean;
    f_AlphaKind: TAlphaKind;
    f_boundthisframe: boolean;
    function _GetIsResident: GLboolean;
    procedure _ReloadID;
  public
    procedure RegisterFields; override;
    procedure BeforeSaving(); override;
    procedure AfterLoading(); override;
    procedure AfterConstruction(); override;
    destructor Destroy; override;
    procedure SaveSelf; override;

    constructor Create;
    procedure Reload; virtual; abstract;
    procedure Bind;
    property IsResident: GLboolean read _GetIsResident;
  end;
  
  TBasicTexture2d = class (TMotherTexture)
    constructor Create(SourceName: AnsiString; HasMips: boolean; ak: TAlphaKind);
    procedure RegisterFields; override;
    procedure Reload; override;
  end;

  TAlphaTexture = class (TMotherTexture)
    constructor Create(SourceName: AnsiString);
    procedure RegisterFields; override;
    procedure Reload; override;
  end;


implementation
  uses
    mo_module;


  type
    TTextureCompressionPolicyDetails = record
      Compress,
      Delete: boolean;
    end;
  const
    TCPD: array[0..MaxTextureTompressionPolicyModel] of TTextureCompressionPolicyDetails
    = ((Compress: No; Delete: No),
       (Compress: Yes; Delete: No),
       (Compress: Yes; Delete: Yes));

  procedure TMotherTexture.RegisterFields;
  begin
    RegType(TypeInfo(TAlphaKind));
    RegSkip( 'f_unclaimed', @f_unclaimed, TypeInfo(boolean));
    RegField('f_sourcename', @f_sourcename, TypeInfo(AnsiString));
    RegField('f_thumb', @f_thumb, TypeInfo(T2DArrayOfDword));
    RegField('f_index', @f_index, TypeInfo(integer));
    RegField('f_Token', @f_Token, TypeInfo(int64));
    RegField('f_thumbToken', @f_thumbToken, TypeInfo(int64));
    RegField('f_name', @f_name, TypeInfo(glUint));
    RegField('f_thumbname', @f_thumbname, TypeInfo(glUint));
    RegField('f_thumbformat', @f_thumbformat, TypeInfo(glInt));
    RegField('f_target', @f_target, TypeInfo(glEnum));
//    RegField('f_quality', @f_quality, TypeInfo(integer));
    RegField('f_priority', @f_priority, TypeInfo(glFloat));
    RegField('f_ReloadUrgency', @f_ReloadUrgency, TypeInfo(float));
    RegField('f_HasMips', @f_HasMips, TypeInfo(boolean));
    RegField('f_AlphaKind', @f_AlphaKind, TypeInfo(TAlphaKind));
    RegField('f_boundthisframe', @f_boundthisframe, TypeInfo(boolean));
  end;
  
  procedure TMotherTexture.SaveSelf;
  var
    TmpThumb: pointer;
  begin
    if (Assigned(f_thumb) or (f_thumbname > 0))
    and not Module.Options.StoreLoResTexturesInSaves then begin
      //temporarily "hide" the thumbnail.
      TmpThumb:=pointer(f_thumb);
      pointer(f_thumb):=nil;
      Inherited;
      pointer(f_thumb):= TmpThumb;
    end
    else Inherited;
  end;

  constructor TMotherTexture.Create;
  begin
    f_index:=Module.TextureIndex.AddRef(Self);
    _ReloadId;
  end;

  procedure TMotherTexture._ReloadID;
  begin
    f_Token:=CgeGenTexture(f_name);
    f_ThumbToken:=CgeGenTexture(f_thumbName);
  end;

  procedure TMotherTexture.BeforeSaving();
  begin
    UnclaimTexture(f_name, f_Token);
    UnclaimTexture(f_thumbname, f_thumbtoken);
  end;

  procedure TMotherTexture.AfterLoading();
  begin
   // if not
    if not ClaimTexture(f_name, f_Token) then begin
      _ReloadId;
      Reload;
    end;
  end;

  procedure TMotherTexture.AfterConstruction();
  begin

  end;
  
  function TMotherTexture._GetIsResident: GLboolean;
  begin
    glAreTexturesResident(1, @f_name, @Result);
  end;
  
  procedure TMotherTexture.Bind;
  begin
    if f_boundthisframe then
      if (f_name > 0) then glBindTexture(f_target, f_name)
    else begin
    
    end;
  end;
  
  destructor TMotherTexture.Destroy;
  begin
    //if texture had been unclaimed, it means we
    //  were saved at least once,
    //so texture should be kept for future use.
    if not f_unclaimed then begin
      CgeDeleteTexture(f_name, f_Token);
      CgeDeleteTexture(f_thumbname, f_thumbtoken);
    end;
    Module.TextureIndex.Release(f_index);
    inherited;
  end;



  procedure TBasicTexture2d.RegisterFields;
  begin
    inherited;
  end;
  
  constructor TBasicTexture2d.Create(SourceName: AnsiString; HasMips: boolean; ak: TAlphaKind);
  begin
    inherited Create;
    f_sourcename:=SourceName;
    f_HasMips:=HasMips;
    f_alphakind:=ak;
    f_target:=GL_TEXTURE_2D;
    Reload;
  end;
  

  procedure TBasicTexture2d.Reload;
  var
    im: TImage;
    InternalFormat, ImageFormat, NearFilter, FarFilter: glInt;
  begin
    im:= TImage.CreateFromFile(f_sourcename);
    im.ConvertToGray;
    Self.Bind;
    glBindTexture(GL_TEXTURE_2D, f_name);

(*
    case f_mmk of
      mmk_None: begin
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
      end;
      mmk_AlphaCorrected: begin
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      end;
      mmk_Normal: begin
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      end;
    end;
    if (Options.TexturesCompressionPolicy > 0)
    and ((im.width >= Options.MinSizeToEmployTextureCompression)
        or (im.height >= Options.MinSizeToEmployTextureCompression))
      then
        Case f_AlphaKind of
          No_Alpha: InternalFormat:=GL_COMPRESSED_RGB_S3TC_DXT1_EXT;
          Threshold_Alpha: InternalFormat:=GL_COMPRESSED_RGBA_S3TC_DXT1_EXT;
          Gradient_Alpha: InternalFormat:=GL_COMPRESSED_RGBA_S3TC_DXT5_EXT;
        end
      else
        Case f_AlphaKind of
          No_Alpha: InternalFormat:=GL_RGB;
          Threshold_Alpha, Gradient_Alpha: InternalFormat:=GL_RGBA;
        end;
*)
      
    glTexImage2D (GL_TEXTURE_2D, 0, GL_ALPHA, im.width, im.height, 0, GL_ALPHA, GL_UNSIGNED_BYTE, im.data);
    im.Free;
  end;


  procedure TAlphaTexture.RegisterFields;
  begin
    inherited;
  end;

  constructor TAlphaTexture.Create(SourceName: AnsiString);
  begin
    inherited Create;
    f_sourcename:= SourceName;
    f_target:= GL_TEXTURE_2D;
    f_HasMips:=No;
    Reload;
  end;
  

  procedure TAlphaTexture.Reload;
  var
    im: TImage;
  begin
    im:= TImage.CreateFromFile(f_sourcename);
    im.ConvertToGray;
    Self.Bind;
    glBindTexture(GL_TEXTURE_2D, f_name);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexImage2D (GL_TEXTURE_2D, 0, GL_ALPHA, im.width, im.height, 0, GL_ALPHA, GL_UNSIGNED_BYTE, im.data);
    im.Free;
  end;



end.
