{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the game module framework class.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

{$ifdef fpc}
  {$mode delphi}
{$endif}
{$longstrings on}
{$minenumsize 4}

unit mo_module;

interface
uses
  un_typedefs, mo_hub, mo_classes, mo_globopts, mo_game, mo_menu, mo_indexer;
  
  function InternalHandleMessage (msg: TCbMessage; p0, p1, p2: integer): boolean;

type
  TKeyboardState = packed array[KEY_WHEEL_DOWN..KEY_WAKE] of longbool;

  TModule = class(TTrulyPersistent)
  protected
    f_basedir,
    f_selfname,
    f_savepath,
    f_sessionpath: Ansistring;
    f_firstRun: boolean;
    f_FrameTime: float;
    function _GetFrameTime: float;
    procedure _LoadPathsFromGlobals;
  public
    Menu: TMenu;
    Game: TGame;
    Options: TGlobalOptions;
    TextureIndex: TIndexer;
    GamePaused: boolean;
    RenderFailed: boolean;
    PrevFrameTime: float;
    mousex, mousey: integer;
    KeyDown: TKeyboardState;
    property BaseDir: AnsiString read f_basedir;
    property SessionPath: AnsiString read f_sessionpath;
    property SelfName: AnsiString read f_SelfName;
    property SavePath: AnsiString read f_SavePath;
    property FrameTime: float read _GetFrameTime;
    constructor Create;
    destructor Destroy; override;
    procedure RegisterFields; override;
    procedure AfterLoading; override;
    procedure BeforeSaving; override;
    function ProcessMessage(msg: TCbMessage; par0, par1, par2: integer): boolean;
    
    procedure OnGetFocus; virtual;
    procedure OnLoseFocus; virtual;
    procedure OnCycle; virtual;
    procedure OnCreate; virtual;
    procedure OnPress(scancode: integer); virtual;
    procedure OnRelease(scancode: integer); virtual;
    procedure OnMouseMove(x, y: integer); virtual;
    
    procedure RenderRenderError; virtual; //if game.render crashed, draw this.
    procedure RenderBackground; virtual; //draw this when there is no game.
  end;
  
  CModule = class of TModule;
  
  procedure CreateModule(c: CModule; SelfName, SessionPath, SavePath: String);
  
var
  Module: TModule;


implementation
  uses sysutils, typinfo, mo_textures;
  
  var G_SelfName, G_SessionPath, G_SavePath: ansistring;

  procedure TModule.RegisterFields;
  begin
    RegClass(TIndexer);
    RegClass(TGame);
    RegClass(TMenu);
    RegClass(TGlobalOptions);
    RegClass(TBasicTexture2d);
    RegType('TKeyboardState', TypeInfo(longbool), SizeOf(TKeyboardState));
    
    RegSkip('f_basedir', @f_basedir, TypeInfo(AnsiString));
    RegSkip('f_selfname', @f_selfname, TypeInfo(AnsiString));
    RegSkip('f_savepath', @f_savepath, TypeInfo(AnsiString));
    RegSkip('f_sessionpath', @f_sessionpath, TypeInfo(AnsiString));

    RegField('f_firstrun', @f_firstrun, TypeInfo(boolean));
    RegSkip ('f_FrameTime', @f_FrameTime, TypeInfo(float)); //no saving amidst the cycle
    
    RegField('Menu', @Menu, TypeInfo(TMenu));
    RegField('Game', @Game, TypeInfo(TGame));
    RegField('Options', @Options, TypeInfo(TGlobalOptions));
    RegField('TextureIndex', @TextureIndex, TypeInfo(TIndexer));
    
    RegField('GamePaused', @GamePaused, TypeInfo(boolean));
    RegField('RenderFailed', @RenderFailed, TypeInfo(boolean));
    RegField('PrevFrameTime', @PrevFrameTime, TypeInfo(float));

    RegField('mousex', @mousex, TypeInfo(integer));
    RegField('mousey', @mousey, TypeInfo(integer));
    RegField('KeyDown', @KeyDown, 'TKeyboardState');
  end;
  
  procedure TModule._LoadPathsFromGlobals;
  begin
    f_basedir:=ExtractFilePath(G_SelfName);
    f_selfname:=ChangeFileExt(ExtractFileName(G_SelfName), '');
    f_savepath:=G_SavePath;
    f_sessionpath:=G_SessionPath;
  end;
  
  procedure TModule.AfterLoading;
  begin
    _LoadPathsFromGlobals;
    DeleteUnclaimedTextures;
  end;
  
  procedure TModule.BeforeSaving;
  begin
    ClearUnclaimedTexturesList;
  end;
  
  Constructor TModule.Create;
  begin
    inherited;
    Module:=Self;
    _LoadPathsFromGlobals;
    f_FirstRun:=Yes;
    Options:=TGlobalOptions.Create;
    TextureIndex:=TIndexer.Create;
  end;
  
  Destructor TModule.Destroy;
  begin
    if Assigned(Game) then Game.Free;
    TextureIndex.Free;
    Options.Free;
    inherited;
  end;
  
  function TModule.ProcessMessage(msg: TCbMessage; par0, par1, par2: integer): boolean;
  begin
    Result:=Yes;
    if f_FirstRun then begin
      OnCreate;
      f_firstRun:=No;
      ClearProcAddresses; //OpenGL;
      InitProcAddresses;//OpenGL;
      OnGetFocus;
    end;
    Case msg of
      Re_OnGetFocus:begin
        ClearProcAddresses; //OpenGL;
        InitProcAddresses;//OpenGL;
        OnGetFocus;
      end;
      Re_OnLoseFocus:
        OnLoseFocus;
      Re_OnMouseMove:
        OnMouseMove(par0, par1);
      Re_OnCycle:
        OnCycle;
      Re_OnPress:
        OnPress(par0);
      Re_OnRelease:
        OnRelease(par0);
    else
      Result:=No;
    end;
  end;

  procedure TModule.OnGetFocus;
  begin
    f_frametime:=0;
  end;
  
  procedure TModule.OnLoseFocus;
  begin

  end;
  

  function TModule._GetFrameTime: float;
  begin
    f_FrameTime:=f_FrameTime + CTimer(); //calling the high-resolution timer
    Result:=f_FrameTime;
  end;
    
  procedure TModule.OnCycle;
  begin
    PrevFrameTime:=FrameTime;
    f_FrameTime:=0.0;
    if Assigned(Menu) then Menu.Cycle;
    if Assigned(Game) and not GamePaused then Game.Cycle(PrevFrameTime);
    if Assigned(Game) then begin
      if not Game.Render then RenderRenderError;
    end
    else RenderBackground;
    if Assigned(Menu) then Menu.Render;
  end;

  procedure TModule.OnCreate;
  begin

  end;

  procedure TModule.OnPress(scancode: integer);
  begin
    KeyDown[scancode]:=Yes;
    if Assigned(Menu)
      then Menu.OnPress(scancode)
      else
        if Assigned(Game) and not GamePaused
          then Game.OnPress(scancode);
  end;

  procedure TModule.OnRelease(scancode: integer);
  begin
    KeyDown[scancode]:=Yes;
    if Assigned(Menu)
      then Menu.OnRelease(scancode)
      else
        if Assigned(Game) and not GamePaused
          then Game.OnRelease(scancode);
  end;

  procedure TModule.OnMouseMove(x, y: integer);
  begin
    if Assigned(Menu)
      then Menu.OnMouseMove(x, y)
      else
        if Assigned(Game) and not GamePaused
          then Game.OnMouseMove(x, y);
  end;
  
  procedure TModule.RenderRenderError;
  begin
  
  end;

  procedure TModule.RenderBackground;
  begin

  end;


var
  FirstRun: boolean = Yes;

  procedure SaveSession;
  var
    f: file;
  begin
    SaveGame(Module,
             MI_OF_SESSION,
             Module.Options.SessionAutosaveFormat,
             Module.SessionPath + Module.SelfName + '.$$$');
    if FileExists(Module.SessionPath + Module.SelfName + '.cge') then begin
      AssignFile(f, Module.SessionPath + Module.SelfName + '.cge');
      Erase(f);
    end;
//    AssignFile(f, Module.SessionPath + Module.SelfName + '.$$$');
    RenameFile(Module.SessionPath + Module.SelfName + '.$$$',
               Module.SessionPath + Module.SelfName + '.cge');
  end;
  
  procedure LoadSession;
  var
    filename: string;
    sig: WideString;
  begin
    sig:=Module.ClassName;
    filename:=Module.SessionPath + Module.SelfName + '.cge';
    Module.Free;
    Module:= LoadGame(
           MI_OF_SESSION, filename, sig) as TModule;
  end;


  procedure CreateModule(c: CModule; SelfName, SessionPath, SavePath: String);
  var
    m: tModule;
    cname, filename: string;
  begin
    G_SelfName:=SelfName;
    G_SessionPath:=SessionPath;
    G_SavePath:=SavePath;
    filename:=SessionPath + ChangeFileExt(ExtractFilename(SelfName), '.cge');
    if CgeFileExists(PAnsiChar(filename))
    then begin
      m:=c.Generate;
      cname:=m.ClassName;
      m.TechnicalDestroy;
      Module:=LoadGame(MI_OF_SESSION, filename, cname) as TModule;
    end
    else Module:=c.Create;
  end;

  function InternalHandleMessage(msg: TCbMessage; p0, p1, p2: integer): boolean;
  var
    s1, s2: AnsiString;
    version: AnsiChar;
  begin
    Try
      Result:=Yes;
{      if FirstRun and FileExists(Module.SessionPath + Module.SelfName + '.cge')
      then begin
        LoadSession;
        FirstRun:=No;
      end;
}
      case msg of
        Re_OnGetFocus: begin
          Result:=Module.ProcessMessage(msg, p0, p1, p2);
        end;
        Re_OnUnload: begin
          SaveSession;
          Module.OnLoseFocus;
        end;
        Re_OnDestroy: begin
          SaveSession;
          Module.Free;
        end;
      {  Re_OnLoad: begin
          ClearProcAddresses; //OpenGL;
          InitProcAddresses;//OpenGL;
          if FileExists(Module.SessionPath + Module.SelfName + '.cge') then LoadSession;
          Module.OnGetFocus;
        end;}
      else
        Result:=Module.ProcessMessage(msg, p0, p1, p2);
      end;
      //raise exception.create('Boo!');

//Function GetTypeData(TypeInfo : PTypeInfo) : PTypeData;
//Function GetEnumName(TypeInfo : PTypeInfo;Value : Integer) : string;
    Except
      s1:=(ExceptObject as Exception).ClassName;
      s2:=(ExceptObject as Exception).Message;
      Die(MI_CRASHED_IN, [GetEnumName(TypeInfo(TCbMessage), Ord(msg)) + '  message handler'
      ,s1, s2]);
    End;
  end;



end.
