{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the container manager.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}


unit cl_container;

interface

uses
  Classes, SysUtils, cl_Dyna;

type

  { TContainer }

  TContainer = class  //(previously descendant of TMemoryStream)
   //note: setting container size erases its contents.
  private
    function GetSize: integer;
  protected
    ChunkSizes: TAOI;
    Chunks: TAOP;
    FName: AnsiString;
    cnum: integer;
  public
    position: integer;
    Constructor Create(Sz: integer);
    Destructor Destroy; override;
    procedure AddChunk(p: pointer; sz: integer); //allocates memory and copies the data
    procedure InsertChunk(p: pointer; sz: integer); //incorporates the already allocated memory with dada in it
    procedure PrepareToRead;
    procedure SetSize(sz: integer);
    procedure Clear();
    procedure Write(var buf; sz: integer);
    procedure Read(var buf; sz: integer);
    function NextChunk(var sz: integer): pointer;
    procedure CopyChunk(var buf; sz: integer);
    function EndOfData(): boolean;
    function MemoryPtr(): pointer; //re-allocates and copies all the data if there is more than one chunk
    procedure WriteToStream(S: TStream);
    procedure LoadFromStream(S: TStream; MaxChunkSize: integer);
    property FileName: AnsiString read FName write FName;
    property Size: integer read GetSize write SetSize;
  end;
  
var
  Containers,
  TexContainer: TStringList;


implementation
  uses cl_hub;
{ TContainer }


  procedure TContainer.Clear();
  var i: integer;
  begin
    For i:=0 to Chunks.High do FreeMem(Chunks[i]);
    Chunks.Length:=0;
    ChunkSizes.Length:=0;
    position:=0;
  end;

  function TContainer.GetSize: integer;
  var
    i: integer;
  begin
    Result:=0;
    For i:=0 to ChunkSizes.High do inc(Result, ChunkSizes[i]);
  end;

  constructor TContainer.Create(Sz: integer);
  var p: pointer;
  begin
    inherited Create;
    Chunks:=TAOP.Create;
    ChunkSizes:=TAOI.Create;
    if Size > 0 then begin
      ChunkSizes.Add(Sz);
      GetMem(p, Sz);
      Chunks.Add(p);
    end;
  end;

  destructor TContainer.Destroy;
  begin
    Clear;
    Chunks.Free;
    ChunkSizes.Free;
    inherited Destroy;
  end;

  procedure TContainer.AddChunk(p: pointer; sz: integer);
  var pm: pointer;
  begin
    GetMem(pm, Sz);
    Chunks.Add(pm);
    ChunkSizes.Add(sz);
    MOVE (p^, Chunks.Last^, sz);
//AddLog('--addchunk %0', [sz]);
  end;

  procedure TContainer.InsertChunk(p: pointer; sz: integer);
  begin
    Chunks.Add(p);
    ChunkSizes.Add(sz);
//AddLog('--insertchunk %0', [sz]);
  end;

  procedure TContainer.PrepareToRead;
  begin
    cnum:=0;
    position:=0;
  end;

  procedure TContainer.SetSize(sz: integer);
  var p: pointer;
  begin
//AddLog('--setsize: chunks %0, size %1, sz %3, name %2', [chunks.length, size, fname, sz]);
    Clear;
    if Sz = 0 then Exit;
    GetMem(p, Sz);
    Chunks.Add(p);
    ChunkSizes.Add(sz);
  end;

  procedure TContainer.Write(var buf; sz: integer);
  begin
    AddChunk(@buf, sz);
  end;

  procedure TContainer.Read(var buf; sz: integer);
  var
    p: pointer;
  begin
    p:=MemoryPtr();//for now, the first call to Read does merge all the chunks.
                   //i'll optimize it later.
    {$ifdef fpc}
    inc(p, position); //E2001 Ordinal type required
    {$else}
    inc(cardinal(p), position);
    {$endif}
    MOVE(p^, buf, sz); //no checking!
    inc(position, sz);
  end;

  function TContainer.NextChunk(var sz: integer): pointer;
  begin
    if not EndOfData then begin
      sz:=ChunkSizes[cnum];
      Result:=Chunks[cnum];
      inc(cnum);
    end
    else begin
      sz:=0;
      Result:=nil;
    end;
//AddLog('--nextchunk %0', [sz]);
  end;

  procedure TContainer.CopyChunk(var buf; sz: integer);
  var stub: integer;
  begin
    MOVE((NextChunk(stub))^, buf, sz);
  end;

  function TContainer.EndOfData(): boolean;
  begin
    Result:= (cnum > Chunks.High);
  end;

  function TContainer.MemoryPtr(): pointer;
  var
    p, p2: pointer;
    s, i: integer;
  begin
//AddLog('--memoryptr: chunks %0, size %1, name %2', [chunks.length, size, fname]);
    if Chunks.Length = 1 then {$ifdef fpc}Exit(Chunks[0]){$else}begin Result:=Chunks[0]; Exit end{$endif};
    if Chunks.Length < 1 then {$ifdef fpc}Exit(NIL){$else}begin Result:=nil; Exit end{$endif};

    //no, there are several chunks.
    // stitch them into one contignuous memory region
    s:=0;
    For i:=0 to ChunkSizes.High do inc(s, ChunkSizes[i]);
    GetMem(p, s);
    p2:=p;
    For i:=0 to ChunkSizes.High do begin
      MOVE(Chunks[i]^, p2^, ChunkSizes[i]);
      {$ifdef fpc}
      inc(p2, ChunkSizes[i]);
      {$else}
      inc(cardinal(p2), ChunkSizes[i]);
      {$endif}
      FreeMem(Chunks[i]);
    end;
    Clear();
    Chunks.Add(p);
    ChunkSizes.Add(s);
    Result:=p;
  end;

  procedure TContainer.WriteToStream(S: TStream);
  var
    i: integer;
  begin
    For i:=0 to Chunks.High do
      S.Write(Chunks[i]^, ChunkSizes[i]);
  end;

  procedure TContainer.LoadFromStream(S: TStream; MaxChunkSize: integer);
  var
    n: integer;
    p: pointer;
  begin
    repeat
      n:=S.size - S.position;
      if (MaxChunkSize > 0) and (n > MaxChunkSize) then n:= MaxChunkSize;
      GetMem(p, n);
      S.Read(p^, n);
      InsertChunk(p, n);
    until S.position = S.size;
  end;

end.

