{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the graphical console class.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}
 unit cl_console;

 interface
 uses
   SysUtils, Classes, un_typedefs, cl_hub;
   
 Const
   ntz = 256; // procedural texture dimensions for the "static" effect.
   
   Type

     { TConsole }

     TConsole = class (TStringList)
     private
       f_texture, f_bsodtexture, f_noisetexture: glUint;
       f_limit,
       f_hidetime,
       f_fadetime,
       f_tostretch: integer;
       f_vis: boolean;
       f_sb: integer;
       f_lasttime: TDateTime;
       f_fadeline: integer;
       f_noisearray: packed array[0..ntz-1, 0..ntz-1] of byte;
       procedure _SetVisible(b: boolean);
     public
       Constructor Create;
       procedure Draw (Clearbackground: boolean);
       procedure Add (S: AnsiString);
       procedure AddComment(S: AnsiString);
       procedure InitGL;
       procedure Delete (i: integer); override;
       property Scrolledback: integer read f_sb write f_sb;
       Property AlwaysVisible: boolean read f_vis write _SetVisible;
     end;
     
 var
   Console: TConsole;
   
   
 Const
   ConsoleCharWidth = 9;
   ConsoleCharHeight = 13;
   ConsoleCharOffsetS = 4;
   ConsoleCharOffsetT = 2;
   ConsoleBorders = 5.0;//0;//5.0;
   
 implementation

   

 { TConsole }

 procedure TConsole._SetVisible(b: boolean);
 begin
   if not b then f_sb:=0;
   f_vis:=b;
 end;

 constructor TConsole.Create;
 begin
   inherited Create;
   f_Limit:=1000;
   Add('    ********  CGE LOG  ********');
 end;
 
 procedure TConsole.InitGL;
 var
   w, h, stub: integer;
   m: TImageMode;
   p: pointer;
 begin
   Try
     f_limit:=Config.IntChk['console','MaxLines',150,10000];
     f_hidetime:=Config['console','MessageDisplayTime'];
     f_fadetime:=Config['console','MessageFadeOutTime'];
     if f_hidetime = 0 then f_hidetime:=1; //division by zero bad
     if f_fadetime = 0 then f_fadetime:=1;
     f_tostretch:=Config.IntChk['console','DoubleSizeIfResolutionHighetThan', 1024, 2048];
     //CreateContainerFromFile('$ct', 'core/data/console.png', stub);
     PrepareToDecodePic ('$console.png', w, h, m);
     if m <> Im_RGBA then Die ('WTF?! console.png is not RGBA!');
     GetMem(p, w * h * ImgModeBpp[m]);
     DecodePic(p);
     DeleteContainer('$console.png');

     glGenTextures(1, @f_texture);
     glBindTexture(GL_TEXTURE_2D, f_texture);
     glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
     glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
     glTexImage2D (GL_TEXTURE_2D, 0, GL_LUMINANCE_ALPHA{GL_RGBA}, w, h, 0, GL_RGBA, GL_UNSIGNED_BYTE, p);
     FreeMem(p);

//     CreateContainerFromFile('$ct', 'core/data/bsod.png', stub);
     PrepareToDecodePic ('$bsod.png', w, h, m);
     if m <> Im_RGB then Die ('WTF?! bsod.png is not RGB!');
     GetMem(p, w * h * ImgModeBpp[m]);
     DecodePic(p);
     DeleteContainer('$bsod.png');

     glGenTextures(1, @f_bsodtexture);
     glBindTexture(GL_TEXTURE_2D, f_bsodtexture);
     glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
     glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
     glTexImage2D (GL_TEXTURE_2D, 0, GL_LUMINANCE, w, h, 0, GL_RGB, GL_UNSIGNED_BYTE, p);
     FreeMem(p);

     
     glGenTextures(1, @f_noisetexture);
   Except
     Die(MI_DIED_INITIALIZING, [RuEn('','console')]);
   End;
 end;


 function CCLd(a: glFloat; b: integer): glFloat;
 begin
   Case b of
     1: if a < 0 then Result:=1.0 else Result:=sqr(1.0 - a);
     2: if a < 0 then Result:=1.0 else Result:=sqr(1.0 - a);
     3: if a < 0 then Result:=1.0 else Result:=sqrt(1.0 - a);
     4: if a < 0 then Result:=1.0 else Result:=(1.0 - a);
   end;
 end;

   function SLines(s:string; _m: integer): integer;
   begin
     Result:=1 + ((Length(s) - 1) div (_m - 1));
   end;

 procedure randoby(var b: byte); //inline;
 var c: integer;
 begin
   c:=Random(101) - 50;
   if (integer (b) + c < 0) or (integer(b) + c > 255)
     then dec(b, c)
     else inc(b, c);
 end;

 procedure TConsole.Draw(Clearbackground: boolean);
 var
   zoom, bsodvoffs: glFloat;
   cuE, y0, lCo, rdh, c, ymax, xmax, Sdh, Swd,  cc, x, y, zms: integer;
   xp, yp, ts, tt, liT: glFloat;
 begin
   zms:= Round((Now() - f_lasttime) * 86400000.0);
   If not (Clearbackground
           or f_vis
           or (zms < (f_hidetime + f_fadetime)))
   then begin
     f_fadeline:= -1;
     Exit;
   end;
   if ClearBackground then f_vis:=Yes;
   
   if f_sb < 0 then f_sb:=0;
   if not f_vis then f_sb:=0;
   
   glEnable(GL_TEXTURE_2D);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
   glEnable(GL_ALPHA_TEST);
   glEnable(GL_BLEND);
   glAlphaFunc(GL_GREATER, 0);
   glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
   glDisable(GL_CULL_FACE);
   glDisable(GL_DEPTH_TEST);
   glViewport(0, 0, TheWindow.DisplayWidth, TheWindow.DisplayHeight);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   glOrtho(0, TheWindow.DisplayWidth, 0, TheWindow.DisplayHeight, 0 ,1);
//   gluOrtho2d(0, TheWindow.DisplayWidth, 0, TheWindow.DisplayHeight);


   If Clearbackground then begin //Big red banner "no modules loaded"
     glColor3f(0, 0, 1);
     if MessageContainer.Language = 'RUSSIAN'
       then bsodvoffs:= 0.5
       else bsodvoffs:= 0;
     glBindTexture(GL_TEXTURE_2D, f_bsodtexture);
     glBegin(GL_QUADS);
       glTexCoord2f(0, 0.5 + bsodvoffs);
       glVertex2f(0, 0); //bottom left corner
       glTexCoord2f(0, 0 + bsodvoffs);
       glVertex2f(0, TheWindow.DisplayHeight);
       glTexCoord2f(1, 0 + bsodvoffs);
       glVertex2f(TheWindow.DisplayWidth, TheWindow.DisplayHeight);
       glTexCoord2f(1, 0.5 + bsodvoffs);
       glVertex2f(TheWindow.DisplayWidth, 0);
     glEnd;


     glBindTexture(GL_TEXTURE_2D, f_noisetexture);
     glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
     glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
     for x:=0 to ntz - 1 do
       for y:=0 to ntz - 1 do
        // f_noisearray[x,y]:= random(256);
         randoby(f_noisearray[x, y]);

     glTexImage2D (GL_TEXTURE_2D, 0, GL_LUMINANCE, ntz, ntz, 0, GL_LUMINANCE, GL_UNSIGNED_BYTE, @f_noisearray[0,0]);
     glNormal3f( 0.0, 0.0, 1.0);
    // glColor3f(0.5, 0.5, 0.35 + 0.35 * sin(Now * 100000.0) / 2);
     glColor4f(1, 1, 1, 0.3);
     glBegin(GL_QUADS);
       glTexCoord2f(0, 2);
       glVertex2f(0, 0); //bottom left corner
       glTexCoord2f(0, 0);
       glVertex2f(0, TheWindow.DisplayHeight);
       glTexCoord2f(1, 0);
       glVertex2f(TheWindow.DisplayWidth, TheWindow.DisplayHeight);
       glTexCoord2f(1, 2);
       glVertex2f(TheWindow.DisplayWidth, 0);
     glEnd;
   end;

   glBindTexture(GL_TEXTURE_2D, f_texture);
   zoom:=1.0;
   Sdh:=TheWindow.DisplayHeight;
   Rdh:=Sdh;
   Swd:=TheWindow.DisplayWidth;

   if TheWindow.DisplayWidth > f_tostretch then zoom:=2.0;// TheWindow.DisplayWidth / f_tostretch;
   xmax :=Trunc( ( (TheWindow.DisplayWidth / zoom) - 2 * ConsoleBorders) / ConsoleCharWidth);
   ymax :=Trunc( ( (TheWindow.DisplayHeight / zoom) - 2 * ConsoleBorders) / ConsoleCharHeight);
//VerboseLog ('xmax=%0, ymax=%1, zoom=%2', [xmax, ymax, zoom]);
   
   if zoom <> 1.0 then begin
     Sdh:=trunc (Sdh / zoom);
     Swd:=trunc (Swd/ zoom);
   end;

   cuE:=Count - 1;
   lCo:=-f_sb;
   while (cuE >= 0)
     and (f_vis or (cuE >= (f_fadeline - 1)))
     and (lCo < ymax)
   do begin
      inc(lCo, SLines(Self.Strings[cuE], xmax));
      dec(cuE);
   end;
   //if CuE < 0 then CuE:=0;
//VerboseLog('cuE=%0, lCo=%1', [cuE, lCo]);
   
   if lCo = 0 then Exit;
   
   if lCo > (ymax{ + f_sb}) then y0:= ymax{ + f_sb} - lCo else y0:=0;

   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

   y:=y0;
   while (cuE < Count - 1) and (y <= ymax) do begin
     x:=0;
     c:=1;
     inc (cuE);

     if f_vis then liT:=0
     else begin
       if zms < f_hidetime then liT:= - (f_hidetime - zms) / f_hidetime
                           else liT:=(zms - f_hidetime) / f_fadetime;
     end;
     glColor4f(CCLd(liT, 1), CCLd(liT, 2), CCLd(liT, 3), CCLd(liT, 4));
     
     While c < (Length(Self.Strings[cuE]) - 1) do begin
       if y > 0 then begin
         //draw character;
         xp:=(ConsoleBorders + (x * ConsoleCharWidth)) * zoom;
         yp:=(ConsoleBorders + (y * ConsoleCharHeight)) * zoom;
         cc:=Ord(Self.Strings[cuE][c]);
         if cc >=32 then begin //ignore special characters
           ts:=((cc mod 16) * 16 + ConsoleCharOffsetS) / 256.0;
           tt:=((cc div 16) * 16 + ConsoleCharOffsetT) / 256.0;
//VerboseLog(chr(c));
           glBegin(GL_QUADS);
             glTexCoord2f(ts, tt); //top left
             glVertex2f(xp, Rdh - yp);
             glTexCoord2f(ts + (ConsoleCharWidth + 1) / 256.0, tt); //top right
             glVertex2f(xp + (ConsoleCharWidth + 1) * zoom, Rdh - yp);
             glTexCoord2f(ts + (ConsoleCharWidth + 1) / 256.0, tt + ConsoleCharHeight / 256.0); //bottom right
             glVertex2f(xp + (ConsoleCharWidth + 1) * zoom, Rdh - yp - ConsoleCharHeight * zoom);
             glTexCoord2f(ts, tt + ConsoleCharHeight / 256.0); //bottom left
             glVertex2f(xp, Rdh - yp - ConsoleCharHeight * zoom);
           glEnd;
         end;
       end;
       inc (c);
       inc (x);
       if x > xmax then begin
         x:=1;
         inc(y);
       end;
     end;
     inc (y);
   end;

   
 end;

 procedure TConsole.Add(s: AnsiString);
 var
   i: integer;
 begin
   //recursion. Scary.
   i:=Pos(#13, s);
   if (i > 0) and (i < Length(s)) then begin
     if i <= 2 then s:= Copy(s, i + 1, Length(s) - i)
     else begin
       s[i]:=#0;//don't draw it
       Add(Copy(s, 1, i));
       Add(Copy(s, i, Length(s) - i + 1));
       Exit;
     end;
   end;
   if (s <> '') and (s[1]<>#0) then s:=#127 + s;
   inherited Add(s);
   if f_fadeline < 0 then f_fadeline:=Count - 1;
   if Count > f_limit then begin
     if f_fadeline > 0 then dec(f_fadeline);
     Delete(0);
   end;
   f_lasttime:=Now();
 end;
 
 procedure TConsole.AddComment(S: AnsiString);
 begin
   if Count = 0 then Add(S)
   else begin
     Strings[Count - 1] := Strings[Count - 1] + S;
     f_lasttime:=Now();
   end;
 end;

 procedure TConsole.Delete(i: integer);
 begin
   inherited Delete(i);
 end;

end.
