{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the module manager.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

{$mode delphi}
{$macro on}
unit _cl_modman;


interface

uses
  SysUtils, un_typedefs, cl_module, typinfo;
Type

  (*

    UPDATE - 09.09.06.
    1. ONLY ONE MODULE CAN BE LOADED AT A TIME.
      Switching to another module causes the current module
      to unload.
    2. DATE IS CHECKED REGARDLESS OF STATE.
      There is no "rollback" now, any error while loading/
      unloading ends with the error screen.
      

    Only one module at a time can have the focus.
    Only the active module is checked for the dll date update,
      and only in the developer mode, and only when the window has focus.
    If the window doresn't have the focus, none of the modules are active.
      

    Developer mode:
    Module #0 is always the dispatcher (like taskbar),
      thus Activate(0) brings up the module selection menu.
      
    User mode:
    There is only one module, #0.
    Usually it's the game module.
  *)
  
  
  { TModuleManager }

  TModuleManager = class
    constructor Create;
    destructor Destroy; OVERRIDE;
  private
    ActiveModuleNum,
    PrevActiveNum: integer;
    function _GetModNum: integer;
  protected
    Module: TModule;
    PrevModuleDateCheckTick, ModuleCheckPeriod: longint;
  public
    Procedure Unload;
    function LoadTheBaseModule(): boolean;
    function Load(FileName: string): boolean;
    function InputMessage (msg: TCbMessage; par0, par1, par2: integer): boolean;
    Procedure Checkdates;
    function Activate: boolean;
    Procedure Deactivate();
 end;

Var
  ModuleManager: TModuleManager;

implementation
  uses cl_hub, cl_console;
  
  constructor TModuleManager.Create;
  var
    i: integer;
  begin
    inherited Create;
    ActiveModuleNum:= -1;
    PrevActiveNum:= 0;
    ModuleCheckPeriod:=Config.IntChk['modules', 'CheckIfModulesAreRecompiledPeriod', 100, 60000];
    if ParamStr(1) <> '' then begin
      RunningOneShot:=Yes;
      Module:=TModule.Create(CgePath(ParamStr(1)));
    end
    else LoadTheBaseModule();
    Activate;
  end;


  
  destructor TModuleManager.Destroy;
  var
    i: integer;
  begin
    try
      if Assigned(Module) then Module.Free
    except
      AddLog(MI_CRASHED);
    end;
    inherited;
  end;


  procedure TModuleManager.Unload;
  var i: integer;
  begin
    Module.Free;
    pointer(Module):=nil;
    ActiveModuleNum:= -1;
    PrevActiveNum:=0;
  end;

  function TModuleManager.LoadTheBaseModule(): boolean;
  begin
    Result:=Load(Config.Path['modules', 'MainModule']);
  end;

  function TModuleManager.Load(FileName: string): boolean;
  var
    i: integer;
    M: TModule;
  begin
    Result:=No;
    if ActiveModuleNum >=0 then Unload;
//    if DeveloperMode then begin
      Try
        M:=TModule.Create(FileName);
      Except
        AddLog(MI_DEVMODE_MODULE_UNLOADED, [StopDying()]);
        ActiveModuleNum:= -1;
        PrevActiveNum:= -1;
        Exit;
      End;
(*
    end
    else
      M:=TModule.Create(FileName); //no exception handling
    *)
    M.Level:=0;
    Module:=M;
    Result:=Yes;
  end;

  function TModuleManager.InputMessage(msg: TCbMessage; par0, par1, par2: integer): boolean;
  begin
    Result:=No;
    Case msg of
      Re_OnGetFocus,
      Re_OnLoseFocus, //Focus is now processed internally
      Re_OnCreate,
      Re_OnDestroy //On Create is processed internally,
                   //On Destroy will cause ModuleManager to unload anyway,
                   //  no need to pass it to the modules.
        : Exit;
    else
      if ActiveModuleNum < 0 then begin
        if TheWindow.Focus then begin
          if PrevActiveNum < 0 then begin
            if msg = Re_OnPress then begin
              if par0 = KEY_ESCAPE then
                if not Activate then begin
                  LoadTheBaseModule();
                  Activate;
                  Console.AlwaysVisible:=No;
                end;
              if par0 = KEY_SPACE then TheWindow.ExitRequested:=Yes;
            end;
            Exit;
          end;
          Activate;
          if ActiveModuleNum < 0 then Exit;
        end
        else begin
          if TheWindow.Focus or not (msg in [Re_OnMouseMove, Re_OnResize, Re_OnPress, Re_OnRelease, Re_OnType, Re_OnCycle])
            then VerboseLog('WARNING! Sending message %0 canceled because there are no active modules.'
                , [GetEnumName(TypeInfo(TCbMessage), Ord(msg))]);
          Exit;
        end;
      end;

      if (msg = Re_OnCycle) and Module.HasFocus and not TheWindow.Focus
      then begin
        Deactivate;
        Exit;
      end;

      //pass it to the active module.
      Result:=Module.InputMessage(msg, par0, par1, par2);
    end;
  end;

  procedure TModuleManager.CheckDates;
  var c: integer;
  begin
    if not (DeveloperMode or DebugMode) then Exit;
    if not TheWindow.Focus then Exit;
    if Self.ActiveModuleNum < 0 then Exit;

    if Tick() < PrevModuleDateCheckTick + ModuleCheckPeriod then Exit;
    
    if not Module.CheckForUpdate
     //crashed loading the backup copy. Remove it.
     then Unload;

    PrevModuleDateCheckTick:=Tick;
  end;

  function TModuleManager.ModuleName(i: integer): string;
  begin
    Result:=Modules[i].FileName;
  end;

  function TModuleManager.Activate(n: integer): boolean;
  begin
    Result:=false;
    if (n = 0) and (NumLevels = 0) then
   //   if (DeveloperMode or DebugMode) then begin
        //There is no module #0 to load
        Exit;
   //   end
   //   else Die('There is no module #0 to load!');
    if ((n >= NumLevels) and (n > 0)) or (n < 0)
      then Die('Attempt to activate module by invalid index #'+ IntToStr(n));
    ActiveModuleNum:=n;
    if not Modules[ActiveModuleNum].HasFocus
      then begin
        Modules[ActiveModuleNum].InputMessage(Re_OnGetFocus, 0,0,0);
      end;
    Result:=True;
  end;
  
  procedure TModuleManager.UnloadActiveModule;
  begin
    Unload(ActiveModuleNum);
    PrevActiveNum:= -1;
  end;

  procedure TModuleManager.Deactivate();
  begin
    if (ActiveModuleNum < 0) or (ActiveModuleNum >= NumLevels) then Exit;
    Modules[ActiveModuleNum].InputMessage(Re_OnLoseFocus, 0,0,0);
    ActiveModuleNum:= -1;
  end;
  

end.

