{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the global options handling routines.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}
{$minenumsize 4}

unit mo_globopts;

interface
  uses un_typedefs, mo_classes;
  
Const
  MaxTextureTompressionPolicyModel = 2;

Type
  TGlobalOptions = class (TTrulyPersistent)
  protected
    f_StoreLoResTexturesInSaves: boolean;
    f_StoreImpostorsInSaves: boolean;
    f_LoResTexturesMipOffset: integer;
    f_TexturesCompressionPolicy: integer;
    f_MinSizeToEmployTextureCompression: integer;
    f_SessionAutosaveFormatIndex: integer;
    
    procedure _SetStoreLoResTexturesInSaves(v: boolean);
    procedure _SetStoreImpostorsInSaves(v: boolean);
    procedure _SetLoResTexturesMipOffset(v: integer);
    procedure _SetTexturesCompressionPolicy(v: integer);
    procedure _SetMinSizeToEmployTextureCompression(v: integer);
    procedure _SetSessionAutosaveFormatIndex(v: integer);
    function _getSessionAutosaveFormat(): AnsiChar;

    function _GetDisplayWidth: integer;
    function _GetDisplayHeight: integer;
//    function _GetDisplayFreq: integer;
    function _GetIsDisplay16bit: boolean;
    function _GetIsFullscreen: boolean;
  public

    property StoreLoResTexturesInSaves: boolean read f_StoreLoResTexturesInSaves write _SetStoreLoResTexturesInSaves;
    property StoreImpostorsInSaves: boolean read f_StoreImpostorsInSaves write _SetStoreImpostorsInSaves;
    property LoResTexturesMipOffset: integer read f_LoResTexturesMipOffset write _SetLoResTexturesMipOffset;
    property TexturesCompressionPolicy: integer read f_TexturesCompressionPolicy write _SetTexturesCompressionPolicy;
    property MinSizeToEmployTextureCompression: integer read f_MinSizeToEmployTextureCompression write _SetMinSizeToEmployTextureCompression;
    property SessionAutosaveFormatIndex: integer read f_SessionAutosaveFormatIndex write _SetSessionAutosaveFormatIndex;
    property SessionAutosaveFormat: AnsiChar read _GetSessionAutosaveFormat;
    
    property FullScreenWidth: integer read _GetDisplayWidth;
    property FullScreenHeight: integer read _GetDisplayHeight;
    property FullScreenIs16Bit: boolean read _GetIsDisplay16bit;
    property IsFullscreenNow: boolean read _GetIsFullscreen;
    
    procedure AfterLoading; override;
    procedure RegisterFields; override;
    Constructor Create;
  end;
  
  





implementation

uses mo_hub;
  
const
  MaxCgeFileFormatIndex = 1;
  CgeFileFormatByIndex: array [0..MaxCgeFileFormatIndex] of AnsiChar = ('0', 's' {,'c'});

  procedure SetConfigBool(s, i: PAnsiChar; v: boolean);
  begin
    if v then _SetConfigInt(s, i, 1) else _SetConfigInt(s, i, 0);
  end;
  
  function GetConfigBool(s, i: PAnsiChar): boolean;
  begin
    Result:=1 = _GetConfigIntCh(s, i, 0, 1);
  end;
  
  Constructor TGlobalOptions.Create;
  begin
    AfterLoading;
  end;

  procedure TGlobalOptions.AfterLoading;
  begin
    f_StoreLoResTexturesInSaves:=GetConfigBool('general', 'store_texture_tumbnails_in_saves');
    f_StoreImpostorsInSaves:=GetConfigBool('general', 'store_impostors_in_saves');
    f_LoResTexturesMipOffset:=_GetConfigIntCh('general', 'tumbnail_is_powers_of_2_smaller', 1, 3);
    f_TexturesCompressionPolicy:=_GetConfigIntCh('video', 'texture_compression_policy', 0, MaxTextureTompressionPolicyModel);
    f_MinSizeToEmployTextureCompression:=_GetConfigIntCh('video', 'min_texture_size_to_compress', 0, 2048);
    f_SessionAutosaveFormatIndex:=_GetConfigIntCh('persistency', 'session_save_format_index', 0, MaxCgeFileFormatIndex);
  end;

  procedure TGlobalOptions.RegisterFields;
  begin
    RegSkip('f_StoreLoResTexturesInSaves', @f_StoreLoResTexturesInSaves, TypeInfo(boolean));
    RegSkip('f_StoreImpostorsInSaves', @f_StoreImpostorsInSaves, TypeInfo(boolean));
    RegSkip('f_LoResTexturesMipOffset', @f_LoResTexturesMipOffset, TypeInfo(integer));
    RegSkip('f_TexturesCompressionPolicy', @f_TexturesCompressionPolicy, TypeInfo(integer));
    RegSkip('f_MinSizeToEmployTextureCompression', @f_MinSizeToEmployTextureCompression, TypeInfo(integer));
    RegSkip('f_SessionAutosaveFormatIndex', @f_SessionAutosaveFormatIndex, TypeInfo(integer));
  end;

  function TGlobalOptions._getSessionAutosaveFormat(): AnsiChar;
  begin
    Result:=CgeFileFormatByIndex[f_SessionAutosaveFormatIndex];
  end;
  
  procedure TGlobalOptions._SetSessionAutosaveFormatIndex(v: integer);
  begin
    f_SessionAutosaveFormatIndex:=v;
    _SetConfigInt('persistency', 'session_save_format_index', v);
  end;

  function TGlobalOptions._GetDisplayWidth: integer;
  begin
    Result:= _GetConfigInt('video', 'width');
  end;
  
  function TGlobalOptions._GetDisplayHeight: integer;
  begin
    Result:= _GetConfigInt('video', 'height');
  end;

  function TGlobalOptions._GetIsDisplay16bit: boolean;
  begin
    Result:= 0 <> _GetConfigInt('video', '16_bit');
  end;

  function TGlobalOptions._GetIsFullscreen: boolean;
  begin
    Result:= 0 <> _GetConfigInt('video', 'fullscreen');
  end;

  procedure TGlobalOptions._SetStoreLoResTexturesInSaves(v: boolean);
  begin
    f_StoreLoResTexturesInSaves:=v;
    SetConfigBool('general', 'store_texture_tumbnails_in_saves', v);
  end;
  
  procedure TGlobalOptions._SetStoreImpostorsInSaves(v: boolean);
  begin
    f_StoreImpostorsInSaves:=v;
    SetConfigBool('general', 'store_impostors_in_saves', v);
  end;
  
  procedure TGlobalOptions._SetLoResTexturesMipOffset(v: integer);
  begin
    f_LoResTexturesMipOffset:=v;
    _SetConfigInt('general', 'tumbnail_is_powers_of_2_smaller', v);
  end;

  procedure TGlobalOptions._SetTexturesCompressionPolicy(v: integer);
  begin
    f_TexturesCompressionPolicy:=v;
    _SetConfigInt('video', 'texture_compression_policy', v);
  end;

  procedure TGlobalOptions._SetMinSizeToEmployTextureCompression(v: integer);
  begin
    f_MinSizeToEmployTextureCompression:=v;
    _SetConfigInt('video', 'min_texture_size_to_compress', v);
  end;
end.
