{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the module manager.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

{$ifdef fpc}
  {$MACRO ON}
  {$MODE Delphi}
  {$longstrings on}
  {$IFDEF Win32}
    {$DEFINE extdecl := stdcall}
  {$ELSE}
    {$DEFINE extdecl := cdecl}
  {$ENDIF}
{$endif}

unit cl_module;


interface

uses
  SysUtils, un_typedefs
  {$ifdef win32}
  ,Windows
  {$else}
  ,dllfuncs, dl
  {$endif};
  
Type

  //TCbMessageProc = function (msg: TCbMessage; par0, par1, par2: integer): boolean; stdcall;

  { TModule }

  TModule = class (TObject)
    constructor Create;
    destructor Destroy; OVERRIDE;
  protected
     _Fname, _FromDir, _ToDir, _Basedir: string;
     DLL: {$IFDEF Win32} THandle {$ELSE} Pointer {$ENDIF} ;
     M_Initialize: procedure(SelfName, SessionsPath, SavePath: PChar); stdcall;
     M_Message: function (msg: TCbMessage; par0, par1, par2: integer): boolean; stdcall;//TCbMessageProc;
     M_SetCallback: procedure(Addr: pointer); stdcall;
     F_Crashed,
     _HadBeenActivated,
     f_ReloadInProgress,
     f_active,
     f_has_focus,
     f_loaded,
     f_notamodule: boolean;
     f_RegetAttempt: integer;
     f_dll_age,
     f_bad_age: longint;
     f_reload_failed_msg: WideString;
     PrevModuleDateCheckTick, ModuleCheckPeriod: longint;
     procedure _LoadAddresses;
     function _GetProcAddress(ProcName: pchar): Pointer;
     function _GetFname: string;
     procedure _CopyModule(l1, l2: string);
     procedure _LoadMeDll(n: string);
     procedure _UnloadMeDll;
     procedure _Initialize;
     function _TryReload: boolean;

  public
    procedure Reload(n: integer);
    property HadBeenActivated: boolean read _HadBeenActivated;
    function InputMessage (msg: TCbMessage; par0, par1, par2: integer): boolean;
    function ProcessMessage (msg: TCbMessage; par0, par1, par2: integer): boolean;
    procedure Load;
    procedure Unload;
    property HasFocus: boolean read f_has_focus write f_has_focus;
    property ErrorMessage: WideString read f_reload_failed_msg;
    property Active: boolean read f_active;
    property Loaded: boolean read f_loaded;
    
    procedure Checkdate;
 end;
 
 
var
  Module: TModule;


implementation
  uses cl_hub, cl_strings, typinfo, cl_console;
  

  procedure TModule.Load;
  begin
    VerboseLog(MI_LOG_ATTACHING_MODULE, [_FName]);

    Try
      if not FileExists(_Fromdir + _FName)
        then Die(MI_ERROR_MODULE_MISSING, [_Fromdir + _FName]);
      f_dll_age:= FileAge(_Fromdir + _FName);
      _CopyModule(_Fromdir + _FName, _ToDir + _FName);
      _LoadMeDll(_ToDir + _Fname);
      _LoadAddresses;
      M_SetCallback(@ExportHostProc);
    Except
      Die(MI_ERROR_CANNOT_LOAD_MODULE, [_Fname]);
    End;
    VerboseLogOk;
  end;
  
  procedure TModule.Reload(n: integer);
  var
    FileName: string;
  begin
    if Loaded then Try
      ProcessMessage(Re_OnUnload,0,0,0);
      Unload;
    Except
      F_Crashed:=Yes;
      F_Loaded:=No;
      AddLog(MI_DEVMODE_MODULE_UNLOADED, [StopDying()]);
      Exit;
    End;

    F_Crashed:=No;

    if n = 0 then begin
      RunningOneShot:=Yes;
      FileName:=OptiFileName(ParamStr(1));
    end
    else begin
      FileName:=OptiFileName(ModulesBasePath + Config.Str['modules', IntToStr(n)]);
      Config.Int['session', 'MostRecentUsedModule']:=n;
    end;
   {$ifdef win32}
     FileName:=StrReplace(FileName, '/', '\');
   {$else}
     FileName:=StrReplace(FileName, '\', '/');
   {$endif}
    _Fname:=ChangeFileExt(ExtractFileName(FileName), DllExtension);
    _Basedir:=ExtractFilePath(FileName);
    _Fromdir:=WorkingDir + _Basedir;

    _ToDir:=OptiPath(Config.Path['modules', 'PutRecompiledModulesTo_' + SystemSuffix]);

    Try
      Load;
      _Initialize;
    Except
      F_Crashed:=Yes;
      AddLog(MI_DEVMODE_MODULE_UNLOADED, [StopDying()]);
    End;
  end;
  
  procedure TModule.Unload;
  begin
    f_loaded:=No;
    _UnloadMeDll;
  end;


  constructor TModule.Create;
  begin
    inherited Create;
    ModuleCheckPeriod:=Config.IntChk['modules', 'CheckIfModulesAreRecompiledPeriod', 100, 60000];
    if ParamStr(1) <> ''
      then Reload(0)
      else Reload(Config.IntChk['session', 'MostRecentUsedModule', 1, 10])
  end;
  
  Procedure TModule._Initialize;
  begin
    VerboseLog(MI_LOG_INITIALIZING_MODULE, [ExtractFileName(_FName)]);
    Try
      M_Initialize(PChar(_Basedir + _FName), PChar(SessionPath), PChar(SavePath));
      VerboseLogOk;
      f_loaded:=Yes;
    Except
      Die(MI_ERROR_MODULE_CRASHED_AT_LOADING, [_FName]);
    End;
  end;
  
  destructor TModule.Destroy;
  begin
    VerboseLog(MI_LOG_UNLOADING_MODULE, [ExtractFileName(_FName)]);
    InputMessage(Re_OnDestroy, 0,0,0);
    _UnloadMeDll;
    VerboseLogOk;
    inherited;
  end;
  
  function TModule._GetProcAddress(ProcName: PChar): Pointer;
  begin
  {$IFDEF Win32}
    Result:=GetProcAddress(DLL, ProcName);
  {$ELSE}
    Result:=dlsym(DLL, ProcName);
  {$ENDIF}
  end;

  Procedure TModule._LoadAddresses;
  begin
    f_notamodule:=Yes; //we loaded it before the linker finished its work...
    M_Initialize:=_GetProcAddress('cgemoduleinit');
    M_Message:= _GetProcAddress('sendmsg');
    M_SetCallback:= _GetProcAddress('setcallback');
    if not Assigned (M_Initialize) or not Assigned (M_Message)
       or not Assigned (M_SetCallback)
       then Die(MI_ERROR_INVALID_MODULE, [_FName]);
    f_notamodule:=No;
  end;
  
   function TModule._TryReload: boolean;
   var
     Crashed, CrashedUnloading, CopiedOver, LoadedDll: boolean;
     DllImage: pointer;
     DllImageSize: integer;
     f: file;
     el_mes: string;
     procedure LoadAndInitMe;
     begin
       Try
        Try
          _LoadMeDll(_ToDir + _Fname);
          _LoadAddresses;
          M_SetCallback(@ExportHostProc);
          LoadedDll:=Yes;
          _Initialize;
        Except
          f_ReloadInProgress:=Yes;
          raise;
        End;
       Except
        Die(MI_ERROR_CANNOT_LOAD_MODULE, [_Fname]);
       End;
     end;
     function TryLoadImage: boolean;
     begin
       el_mes:='';
       Try
         AssignFile(f, _Fromdir + _FName);
         Reset(f, 1);
         DllImageSize:=FileSize(f);
         if DllImageSize = 0 then raise Exception.Create('Zero DLL file size -- possibly, compilation still in progress.');
         GetMem(DllImage, DllImageSize);
         BlockRead(f, DllImage^, DllImageSize);
         Close(f);
         Result:=Yes;
       Except
         if ExceptObject is Exception
           then el_mes:=(ExceptObject as Exception).Message;
         Result:=No;
         Try Close(f) Except End;
       End;
     end;
     procedure WriteImage;
     begin
       AssignFile(f, _ToDir + _FName);
       Rewrite(f, 1);
       BlockWrite(f, DllImage^, DllImageSize);
       Close(f);
       FreeMem(DllImage);
     end;
   begin
     Result:=Yes;
     if not f_ReloadInProgress then begin
       f_RegetAttempt:=0;
     end;
     VerboseLog('Trying to replace the module %0 with the new, recompiled version...',[_FName]);
     f_ReloadInProgress:=Yes;

     if not TryLoadImage then begin
       inc(f_RegetAttempt);
       VerboseLog('Attempt #%0 to read "%1" failed with "%2".'
                  , [f_RegetAttempt, _Fromdir + _FName, el_mes]);
       Exit;
     end;
      
     f_ReloadInProgress:=No;
     
     Try
       Crashed:=No;
       CrashedUnloading:=Yes;
       f_dll_age:=0;
       CopiedOver:=No;
       LoadedDll:=No;
       InputMessage(Re_OnUnload, 0,0,0);
       _UnloadMeDll;
       CrashedUnloading:=No;
       _CopyModule(_ToDir + _FName, _ToDir + ChangeFileExt(_FName, '.back'));
       WriteImage;
       CopiedOver:=Yes;
       LoadAndInitMe;
       f_dll_age:= FileAge(_Fromdir + _FName);
     Except
       if CrashedUnloading then Die(MI_CRASHED_UNLOADING_MODULE, [_Fname]);
       Crashed:=Yes;
       f_reload_failed_msg:=StopDying();//PullWarningsFromQueue;
       Try
         f_bad_age:=FileAge(_Fromdir + _FName);
       Except
         f_bad_age:=0;
       End;
       if f_dll_age <> 0 then f_bad_age:=f_dll_age;
//       if not f_notamodule then begin
         //this was a real error. stop and display it.
         AddLog(RuEn(' . %0','Attempt failed. %0'), [f_reload_failed_msg]);
         Result:=No;
         Exit;
(*       end
       else begin
         //this was just a glitch: reload the backup copy and try again later.
         VerboseLog(RuEn('    .  .','Crashed reloading the new module. Switching to the backup copy.'));
        {$ifdef win32}
         SetLastError(0);
         Sleep(100);
        {$endif}
         if LoadedDll then Try
            _UnloadMeDll;
            SetLastError(0);
           Except
             VerboseLog('Failed to unload the DLL. Error message is "%0"', [StopDying()]);
           End;
         IOResult;
         if DebugMode then begin
           Result:=No;
           Exit;
         end;
         if CopiedOver then begin
           VerboseLog('Restoring the module from the backup copy...');
           _CopyModule(_ToDir + ChangeFileExt(_FName, '.back'), _ToDir + _FName);
           VerboseLog('Ok.')
         end;
         Try
           LoadAndInitMe;
           F_Crashed:=No;
           InputMessage(Re_OnLoad, 0,0,0);
           Result:=Yes;
           Exit;
         Except
           Try
             Die(MI_CRASHED_RESTORING_MODULE_FROM_BACKUP, [_Fname, ExtractDyingYell()]);
           Except
             AddLog(StopDying());
           End;
           F_Crashed:=Yes;
           Result:=No;
         End;
       end;*)
     End;
     if Result then begin
       f_Crashed:=false;
       f_loaded:=true;
       InputMessage(Re_OnLoad, 0,0,0);
     end;
   end;

   function TModule._GetFname;
   begin
     Result:=_ToDir + _FName;
   end;

   procedure TModule._CopyModule(l1, l2: string);
   var
     f: file;
     p: pointer;
     s: integer;
   begin
//     VerboseLog(#10#13'Copying file "%0" to "%1"', [l1,l2]);
      Try
        if FileExists(l2) then Try
          AssignFile(f, l2);
          Erase(f);
        Except
          Die(MI_ERROR_CANNOT_DESTROY_FILE,[l2]);
        End;

        AssignFile(f, l1);
        Reset(f, 1);
        s:=FileSize(f);
        GetMem(p, s);
        BlockRead(f, p^, s);
        Close(f);
        AssignFile(f, l2);
        Rewrite(f, 1);
        BlockWrite(f, p^, s);
        Close(f);
        FreeMem(p);
      Except
        Die(MI_CANNOT_COPY_MODULE_DLL, [l1, l2]);
      End;
   end;

   procedure TModule._LoadMeDll(n: string);
   begin
     f_notamodule:=Yes;
      {$IFDEF Win32}
        SetLastError(0);
        DLL:= LoadLibrary(PChar(n));
        if DLL = 0 then Exception.Create('LoadLibrary() returned zero.');
      {$ELSE}
        DLL:= dlopen(PChar(n), RTLD_NOW);
        If not Assigned(DLL) then Raise Exception.Create('dlopen() returned NULL.');
      {$ENDIF}
     f_notamodule:=No;
   end;

   procedure TModule._UnloadMeDll;
   begin
    f_loaded:=No;
    {$IFDEF Win32}
    FreeLibrary(DLL);
    {$ELSE}
    dlClose(DLL);
    {$ENDIF}
   end;

   function TModule.InputMessage (msg: TCbMessage; par0, par1, par2: integer): boolean;
   begin
    Result:=No;
    Case msg of
      Re_OnGetFocus,
      Re_OnLoseFocus, //Focus is now processed internally
      Re_OnCreate //On Create is processed internally
        : Exit;
    else
      if not f_loaded then begin
        if (msg = Re_OnPress) and (par0 = KEY_SPACE) then begin
          _TryReload;
          if f_loaded then Console.AlwaysVisible:=No;
        end;
        if (msg = Re_OnPress) and (par0 = KEY_ESCAPE)
          then TheWindow.ExitRequested:=Yes;
        Exit;
      end;

      if (msg = Re_OnCycle) and Self.HasFocus and not TheWindow.Focus
      then begin
        Self.ProcessMessage(Re_OnLoseFocus, 0,0,0);
        Exit;
      end;

      Result:=Self.ProcessMessage(msg, par0, par1, par2);
    end;
  end;


   function TModule.ProcessMessage (msg: TCbMessage; par0, par1, par2: integer): boolean;
   begin
     result:=No;
     
     if F_Crashed then Exit;
     if not f_loaded then Exit;

     case msg of
       Re_OnGetFocus: f_has_focus:=Yes;
       Re_OnLoseFocus: f_has_focus:=No;
     end;
     

     Try
       Result:=M_Message(msg, par0, par1, par2);
     Except
       F_Crashed:=Yes;
       Try
         Die(MI_ERROR_MODULE_CRASHED, [Self._FName, GetEnumName(TypeInfo(TCbMessage), Ord(msg))]);
       Except
       End;
       Try
         Unload;
       Except
       End;
       AddLog(MI_DEVMODE_MODULE_UNLOADED, [StopDying()]);
     End;

   end;
   
  procedure TModule.Checkdate;
  var fa: longint;
  begin
    if Tick() < PrevModuleDateCheckTick + ModuleCheckPeriod then Exit;
    fa:=FileAge(_Fromdir + _FName);
//VerboseLog('Age %0; New age %1;  Bad age %2', [f_dll_age, fa, f_bad_age]);
    if (fa > 0) and (fa <> f_dll_age) and (fa <> f_bad_age)
      then _TryReload();
    PrevModuleDateCheckTick:=Tick;
  end;

end.

