{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the module manager.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

{$MACRO ON}
{$MODE Delphi}
{$IFDEF Win32}
  {$DEFINE extdecl := stdcall}
{$ELSE}
  {$DEFINE extdecl := cdecl}
{$ENDIF}

unit cl_module;


interface

uses
  SysUtils, cl_typedefs
  {$ifdef win32}
  ,Windows
  {$else}
  ,dllfuncs, dl
  {$endif};
  
Type
  TCbMessageProc = function (msg: TCbMessage; par: array of integer): boolean; extdecl;

  { TModule }

  TModule = class
    constructor Create(Level: integer; FileName: String);
    destructor Destroy; OVERRIDE;
  protected
     _Fname, _FromDir, _ToDir: string;
     f_level: integer;
     DLL: {$IFDEF Win32} THandle {$ELSE} Pointer {$ENDIF} ;
     M_Initialize: procedure; extdecl;
     M_Message: TCbMessageProc;
     M_SetCallback: procedure(Addr: pointer); extdecl;
     F_Crashed,
     _HadBeenActivated,
     f_ReloadInProgress,
     f_active,
     f_had_ongetfocus,
     f_loaded: boolean;
     f_RegetAttempt: integer;
     f_dll_age,
     f_bad_age: longint;
     f_reload_failed_msg: WideString;
     procedure _LoadAddresses;
     function _GetProcAddress(ProcName: String): Pointer;
     function _GetFname: string;
     procedure _CopyModule(l1, l2: string);
     procedure _LoadMeDll(n: string);
     procedure _UnloadMeDll;
     
  public
    property HadBeenActivated: boolean read _HadBeenActivated;
    function InputMessage (msg: TCbMessage; par: array of integer): boolean;
    procedure Initialize(num: integer);
    procedure CheckForUpdate;
    procedure TryReload;
    property FileName: string read _GetFname;
    property Level: integer read f_level write f_level;
    property HadFocus: boolean read f_had_ongetfocus write f_had_ongetfocus;
    property ErrorMessage: WideString read f_reload_failed_msg;
    property Active: boolean read f_active;
    property Loaded: boolean read f_loaded;
 end;


implementation
  uses cl_hub, cl_strings, cl_modman, typinfo;
  
  constructor TModule.Create(Level: integer; FileName: String);
  begin
    inherited Create;
    _Fname:=ChangeFileExt(ExtractFileName(FileName), DllExtension);
    _Fromdir:=OptiPath(ExtractFilePath(FileName));
    _ToDir:=OptiPath(Config.Path['modules', 'PutRecompiledModulesTo_' + SystemSuffix]);

    AddLog(MI_LOG_ATTACHING_MODULE, [_FName]);

    Try
      if not FileExists(_Fromdir + _FName)
        then Die(MI_ERROR_MODULE_MISSING, [_Fromdir + _FName]);
      f_dll_age:= FileAge(_Fromdir + _FName);
      _CopyModule(_Fromdir + _FName, _ToDir + _FName);
      _LoadMeDll(_ToDir + _Fname);
      _LoadAddresses;
      M_SetCallback(@ExportHostProc);
    Except
      Die(MI_ERROR_CANNOT_LOAD_MODULE, [_Fname]);
    End;
    AddLogOk;
    
    Initialize(Level);
  end;
  
  Procedure TModule.Initialize(num: integer);
  begin
    f_level:=num;
    AddLog(MI_LOG_INITIALIZING_MODULE, [ExtractFileName(_FName)]);
    Try
      M_Initialize;
    Except
      Die(MI_ERROR_MODULE_CRASHED_AT_LOADING, [_FName]);
    End;
    AddLogOk;
    f_loaded:=Yes;
  end;
  
  destructor TModule.Destroy;
  begin
    AddLog(MI_LOG_UNLOADING_MODULE, [ExtractFileName(_FName)]);
    InputMessage(Re_OnDestroy, []);
    _UnloadMeDll;
    AddLogOk;
    inherited;
  end;
  
  function TModule._GetProcAddress(ProcName: String): Pointer;
  begin
  {$IFDEF Win32}
    Result:=GetProcAddress(DLL, PChar(ProcName));
  {$ELSE}
    Result:=dlsym(DLL, PChar(ProcName));
  {$ENDIF}
  end;

  Procedure TModule._LoadAddresses;
  begin
    M_Initialize:=_GetProcAddress('CGEMODULEINIT');
    M_Message:= _GetProcAddress('SENDMSG');
    M_SetCallback:= _GetProcAddress('SETCALLBACK');
    if not Assigned (M_Initialize) or not Assigned (M_Message)
       or not Assigned (M_SetCallback)
       then Die(MI_ERROR_INVALID_MODULE, [_FName]);
  end;
  
   procedure TModule.CheckForUpdate;
   var fa: longint;
   begin
     fa:=FileAge(_Fromdir + _FName);
//VerboseLog('Age %0; New age %1;  Bad age %2', [f_dll_age, fa, f_bad_age]);
     if  f_ReloadInProgress
       or ((fa > 0) and (fa <> f_dll_age) and (fa <> f_bad_age))
       then TryReload;
   end;




   procedure TModule.TryReload;
   var
     Crashed, CopiedOver, LoadedDll: boolean;
     DllImage: pointer;
     DllImageSize: integer;
     f: file;
     el_mes: string;
     procedure LoadAndInitMe;
     begin
       Try
        Try
          _LoadMeDll(_ToDir + _Fname);
          _LoadAddresses;
          M_SetCallback(@ExportHostProc);
          LoadedDll:=Yes;
          Initialize(f_level);
        Except
          f_ReloadInProgress:=Yes;
          raise;
        End;
        InputMessage(Re_OnCreate, []);
        InputMessage(Re_OnGetFocus, []);
        if Crashed then InputMessage(Re_OnReloadEnd, [1]) else InputMessage(Re_OnReloadEnd, [0]);
       Except
        Die(MI_ERROR_CANNOT_LOAD_MODULE, [_Fname]);
       End;
     end;
     function TryLoadImage: boolean;
     begin
       el_mes:='';
       Try
         AssignFile(f, _Fromdir + _FName);
         Reset(f, 1);
         DllImageSize:=FileSize(f);
         if DllImageSize = 0 then raise Exception.Create('Zero DLL file size -- possibly, compilation still in progress.');
         GetMem(DllImage, DllImageSize);
         BlockRead(f, DllImage^, DllImageSize);
         Close(f);
         Result:=Yes;
       Except
         if ExceptObject is Exception
           then el_mes:=(ExceptObject as Exception).Message;
         Result:=No;
         Try Close(f) Except End;
       End;
     end;
     procedure WriteImage;
     begin
       AssignFile(f, _ToDir + _FName);
       Rewrite(f, 1);
       BlockWrite(f, DllImage^, DllImageSize);
       Close(f);
       FreeMem(DllImage);
     end;
   begin
     if not f_ReloadInProgress then begin
       f_RegetAttempt:=0;
     end;
     VerboseLog('Trying to replace the module %0 with the new, recompiled version...',[_FName]);
     f_ReloadInProgress:=Yes;

     if not TryLoadImage then begin
       inc(f_RegetAttempt);
       VerboseLog('Attempt #%0 to read "%1" failed with "%2".'
                  , [f_RegetAttempt, _Fromdir + _FName, el_mes]);
       Exit;
     end;
      
     f_ReloadInProgress:=No;
     
     Try
       InputMessage(Re_OnReloadStart, []);
       _UnloadMeDll;
       Crashed:=No;
       CopiedOver:=No;
       LoadedDll:=No;
       _CopyModule(_ToDir + _FName, _ToDir + ChangeFileExt(_FName, '.back'));
       WriteImage;
       CopiedOver:=Yes;
       LoadAndInitMe;
       f_dll_age:= FileAge(_Fromdir + _FName);
     Except
       Crashed:=Yes;
       f_reload_failed_msg:=PullWarningsFromQueue;
       Try
         f_bad_age:=FileAge(_Fromdir + _FName);
       Except
         f_bad_age:=0;
       End;
       VerboseLog('Crashed reloading the new module. Switching to the backup copy.');
      {$ifdef win32}
       SetLastError(0);
       Sleep(100);
      {$endif}
       if LoadedDll then Try
          _UnloadMeDll;
          SetLastError(0);
         Except
           VerboseLog('Failed to unload the DLL. Error message is "%0"', [ToldException(ExceptObject as Exception)]);
         End;
       IOResult;
       if CopiedOver then begin
         VerboseLog('Restoring the module from the backup copy...');
         _CopyModule(_ToDir + ChangeFileExt(_FName, '.back'), _ToDir + _FName);
         VerboseLog('Ok.')
       end;
       LoadAndInitMe;
     End;
   end;

   function TModule._GetFname;
   begin
     Result:=_ToDir + _FName;
   end;

   procedure TModule._CopyModule(l1, l2: string);
   var
     f: file;
     p: pointer;
     s: integer;
   begin
//     VerboseLog(#10#13'Copying file "%0" to "%1"', [l1,l2]);
      Try
        if FileExists(l2) then Try
          AssignFile(f, l2);
          Erase(f);
        Except
          Die(MI_ERROR_CANNOT_DESTROY_FILE,[l2]);
        End;

        AssignFile(f, l1);
        Reset(f, 1);
        s:=FileSize(f);
        GetMem(p, s);
        BlockRead(f, p^, s);
        Close(f);
        AssignFile(f, l2);
        Rewrite(f, 1);
        BlockWrite(f, p^, s);
        Close(f);
        FreeMem(p);
      Except
        Die(MI_CANNOT_COPY_MODULE_DLL, [l1, l2]);
      End;
   end;

   procedure TModule._LoadMeDll(n: string);
   begin
      {$IFDEF Win32}
        DLL:= LoadLibrary(PChar(n));
        if DLL = 0 then Raise Exception.Create('LoadLibrary() returned zero.');
      {$ELSE}
        DLL:= dlopen(PChar(n), RTLD_NOW);
        If not Assigned(DLL) then Raise Exception.Create('dlopen() returned NULL.');
      {$ENDIF}
   end;

   procedure TModule._UnloadMeDll;
   begin
    f_loaded:=No;
    {$IFDEF Win32}
    FreeLibrary(DLL);
    {$ELSE}
    dlClose(DLL);
    {$ENDIF}
   end;

   function TModule.InputMessage (msg: TCbMessage; par: array of integer): boolean;
   begin
     if F_Crashed then begin
       verboselog('WARNING! Sending message ' + GetEnumName(TypeInfo(TCbMessage), Ord(msg)) + ' canceled because module ' + FileName + ' had previously crashed.');
       Exit;
     end;
     if not f_loaded then begin
       verboselog('NOTE: Sending message ' + GetEnumName(TypeInfo(TCbMessage), Ord(msg)) + ' canceled because module ' + FileName + ' is currently unloaded.');
       Exit;
     end;
     
     Try
       Result:=M_Message(msg, par);
     Except
       F_Crashed:=Yes;
       Die(MI_ERROR_MODULE_CRASHED, [ExtractFileName(FileName)]);
     End;
   end;

end.

