{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains the module manager.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

{$mode delphi}
{$macro on}
unit cl_modman;


interface

uses
  SysUtils, cl_typedefs, cl_module, typinfo;
Type

  { TModuleManager }

  TModuleManager = class
    constructor Create;
    destructor Destroy; OVERRIDE;
  private
    function _GetModNum: integer;
  protected
    Modules: array of TModule;
    PrevModuleDateCheckTick, ModuleCheckPeriod: longint;
  public
    Procedure Unload(level: integer);
    function Load(level: integer; FileName: string): boolean;
    function InputMessage (msg: TCbMessage; par: array of integer): boolean;
    Procedure Checkdates;
    Property NumLevels: integer read _GetModNum;
    Function ModuleName(i: integer): string;
 end;

Var
  ModuleManager: TModuleManager;

implementation
  uses cl_hub;
  
  constructor TModuleManager.Create;
  var
    i: integer;
  begin
    inherited Create;
    ModuleCheckPeriod:=Config.IntChk['modules', 'CheckIfModulesAreRecompiledPeriod', 100, 60000];
    SetLength(Modules, 1);
    if ParamStr(1) <> '' then begin
      RunningOneShot:=Yes;
      Modules[0]:=TModule.Create(0, CgePath(ParamStr(1)));
    end
    else Modules[0]:=TModule.Create(0, Config.Path['modules', 'MainModule']);
  end;


  
  destructor TModuleManager.Destroy;
  var
    i: integer;
  begin
    For i:=Length(Modules)-1 downto 0 do try Modules[i].Free except AddLog(MI_CRASHED); end;
    inherited;
  end;

  function TModuleManager._GetModNum: integer;
  begin
    Result:=Length(Modules);
  end;

  procedure TModuleManager.Unload(level: integer);
  var i: integer;
  begin
    Modules[level].Free;
    For i:=level to Length(Modules) - 2 do begin
      Modules[i]:=Modules[i + 1];
      Modules[i].Level:=i;
    end;
    SetLength(Modules, Length(Modules) - 1);
  end;

  function TModuleManager.Load(level: integer; FileName: string): boolean;
  var
    i: integer;
    M: TModule;
  begin
    if Level > Length (Modules) then begin
      VerboseLog('WARNING! Attempt to load module %0 at level %1 while only %2 levels available!', [FileName, Level, Length(Modules)]);
      Level:=Length(Modules);
    end;
    Try
      M:=TModule.Create(level, FileName);
    Except
      Result:=No;
    End;
    SetLength(Modules, Length(Modules) + 1);
    For i:=Length(Modules) - 2 downto i do begin
      Modules[i + 1]:=Modules[i];
      Modules[i + 1].Level:=i + 1;
    end;
    Modules[i]:=M;
    Result:=Yes;
  end;

  function TModuleManager.InputMessage(msg: TCbMessage; par: array of integer): boolean;
  var
    i: integer;
  begin
    i:=0;
    Result:=No;
    repeat
      if Modules[i].Loaded then begin
        if not Modules[i].HadFocus and not (msg in [Re_OnCreate, Re_OnDestroy, Re_OnGetFocus])
          and assigned(TheWindow)
          then
            if not Modules[i].InputMessage(Re_OnGetFocus, []) then begin
              VerboseLog('WARNING! Sending message %0 canceled because module #%1 didn''t acknowledge receiving the additional Re_OnGetFocus message.'
                , [GetEnumName(TypeInfo(TCbMessage), Ord(msg)), i]);
              inc(i);
              Continue;
            end;
        Result:=Modules[i].InputMessage(msg, par);
        if (msg = Re_OnGetFocus) and Result
          then Modules[i].HadFocus:=True;
      end;
      inc(i);
    until (Result) or (i >= Length(Modules));
  end;

  procedure TModuleManager.CheckDates;
  var c: integer;
  begin
   if Tick() > PrevModuleDateCheckTick + ModuleCheckPeriod then begin
     For c:=0 to High(Modules) do
       Modules[c].CheckForUpdate;
     PrevModuleDateCheckTick:=Tick;
   end;
  end;

  function TModuleManager.ModuleName(i: integer): string;
  begin
    Result:=Modules[i].FileName;
  end;
  

end.

