{
    This file is part of the Cheb's Game Engine,
    Copyright (c) 2004-2006 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains configuration manager class.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}

 {$define cl_objecttype}
 {$define typeofit := TMemIniFile}
 {$define typename := TAoIni}
  {$include cl_dyna_1template.inc}
 {$undef cl_objecttype}

 function LimitsValid(sect, id: string; val: integer): boolean;
 var
   j: integer;
   u: string;
 begin
   Result:= Yes;
   u:=UpperCase(sect + '.' + id);
   For j:=0 to High(ConfigLimits) do
     with ConfigLimits[j] do
       if (s = u) and ((val < min) or (val > max)) then begin
         Result:= No;
         Exit;
       end;
 end;

 function TConfigManager._Find(s, i: string): string;
 var
   j: integer;
   bs, fs: string;
 begin
   s:=UpperCase(s);
   bs:=Ini[0].ReadString(s, i, '');
   wherefound:=0;
   result:=bs;
   For j:=1 to Ini.High do begin
     fs:=Ini[j].ReadString(s, i, '');
     if fs <> '' then begin
       if (bs<>'') and ((Pos(s, overlist[j]) = 0) or (s='MAIN')) then
         VerboseLog('The config file "' + Ini[j].FileName +'" has no rights to override "' + i + '" parameter of "' + s + '" section with value "' + fs + '".')
       else begin
         Result:=fs;
         wherefound:=j;
       end;
     end;
   end;
 end;
 
 function TConfigManager._FindWritePos(s, i: string): integer;
 var
   j: integer;
   bs, fs: string;
 begin
   s:=UpperCase(s);
   result:=1;
   bs:=Ini[0].ReadString(s, i, '');
   if bs<>'' then Result:=0;
   For j:=1 to Ini.High do begin
     fs:=Ini[j].ReadString(s, i, '');
     if (j=1) or (fs<>'') or (Pos(s, UpperCase(Ini[j].ReadString('CONFIG', 'WRITE_SECTIONS', ''))) > 0) then begin
       if (bs<>'') and ((Pos(s, overlist[j]) = 0) or (s='MAIN')) then
         VerboseLog('The config file "' + Ini[j].FileName +'" has no rights to override "' + i + '" parameter of "' + s + '" section with value "' + fs + '".')
       else Result:=j;
     end;
   end;
 end;

 function TConfigManager._ReadStr(s, i: string): string;
 begin
   if UpperCase(s) = 'MAIN' //cannot be overridden
     then Result:=Ini[0].ReadString(s, i, '')
     else Result:=_Find(UpperCase(s), UpperCase(i));
 end;
 
 function TConfigManager._ReadPath(s, i: string): string;
 begin
   Result:=_ReadStr(s, i);
   if  (Result = '') or ({$ifdef win32} ExtractFileDrive(Result) = ''{$else} Result[1] <> '/'{$endif})
   then Result:=WorkingDir + Result;
   Result:=OptiPath(StrReplace(Result, '{$PLATFORM}', SystemSuffix)) + ExtractFileName(Result);
 end;
 
 procedure TConfigManager._WritePath(s,i, d: string);
 begin
   _WriteStr(OptiPath(s), i, d);
 end;

 function TConfigManager._ReadBool(s, i: string): boolean;
 var j: integer;
 begin
   if StrictLimitsCheck then j:=_ReadInt(s, i, 0, 1)
                        else j:=_ReadInt(s, i);
   Result:=(j = 1);
 end;

 function TConfigManager._ReadInt(s, i: string): integer;
 var v: string;
 begin
   v:=_ReadStr(s, i);
   if v='' then v:='0';
   Try
     Result:=StrToInt(v);
   Except
     if MCInitialized then Die (MI_ERROR_INVALID_CONFIG_PARAMETER_NO_LIM, [Ini[wherefound].FileName, s, i, v])
     else raise Exception.Create(Format(
       PIYConfManCrash, [Ini[wherefound].FileName, (ExceptObject as Exception).Message]));
   End;
 end;
 
 function TConfigManager._ReadIntChk(s, i: string; min,max: integer): integer;
 begin
   Result:=_ReadInt(s, i);
   if (Result < min) or (Result > max) then
     ScreamLimitsRead(wherefound, s, i, Result, min, max);
 end;

 function TConfigManager._ReadInt(s, i: string; min, max: integer): integer;
 var v: string;
 begin
   if not StrictLimitsCheck then begin
     Result:=_ReadInt(s, i);
     if Result < min then Result:=min;
     if Result > max then Result:=max;
   end
   else begin
     v:=_ReadStr(s, i);
     if v='' then v:='0';
     Try
       Result:=StrToInt(v);
       if (Result < min) or (Result > max) then raise Exception.Create('Value is out ot bounds!');
     Except
       if MCInitialized then Die (MI_ERROR_INVALID_CONFIG_PARAMETER, [Ini[wherefound].FileName, s, i, v, min, max])
       else raise Exception.Create(Format(
         PIYConfManCrash, [Ini[wherefound].FileName, (ExceptObject as Exception).Message]));
     End;
   end;
 end;

 procedure TConfigManager._WriteStr(s,i,d: string);
 var p:integer;
 begin
   p:=_FindWritePos(s, i);
   if p = 0
     then Die(MI_ERROR_MAININI_READ_ONLY, [s, i, d])
     else Ini[p].WriteString(s, i, d);
 end;

 procedure TConfigManager._WriteInt(s,i: string; v:integer);
 begin
   _WriteStr(s, i, IntToStr(v));
 end;
 
 procedure TConfigManager._WriteIntChk(s,i: string; min,max,v:integer);
 begin
   if (v < min) or (v > max) then begin
     VerboseLog('Attempt to write invalid configuration data. Section ='+s+' par='+i+' min='+IntToStr(min)+' max='+IntToStr(max)+' value='+IntToStr(v)+'. Value truncated.');
   end;
   if v < min then v:=min;
   if v > max then v:=max;
   _WriteStr(s, i, IntToStr(v));
 end;

 procedure TConfigManager._WriteBool(s,i: string; b: boolean);
 begin
   if b then _WriteStr(s, i, '1') else _WriteStr(s, i, '0');
 end;

 function TConfigManager.AttachIni(FileName: string): boolean;
 var j: integer;
 begin
   Result:=No;
   For j:=1 to Ini.High do
     if UpperCase(Ini[j].FileName) = UpperCase(FileName) then Exit;
   if not MCInitialized and not FileExists(FileName)
     then raise Exception.Create(Format(PIYConfMissing, [FileName]));
   Try
     Ini.Add(TMemIniFile.Create(FileName));
   Except
     if MCInitialized then Die (MI_ERROR_ACCESSING , [FileName])
     else raise Exception.Create(Format(
       PIYConfManCrash, [FileName, (ExceptObject as Exception).Message]));
   End;
   overlist.Length:=Ini.Length;
   VerboseLog(' %0 attached at pos %1.',[Ini.Last.FileName, Ini.High]);
   Result:=Yes;
 end;

 Constructor TConfigManager.Create(name: string);
 begin
   _filename:=name;
   SilentCorrectionsOnWrite:=No;
   Ini:=TAoIni.Create;
   overlist:=TAOS.Create;
   overlist.add('');
   Ini.Container:=Yes;
   AttachIni(_filename);
 end;

 Destructor TConfigManager.Destroy;
 var j: integer;
 begin
   For j:=1 to Ini.High do Ini[j].UpdateFile;
   Ini.Free;
   overlist.Free;
 end;

 procedure TConfigManager.AttachOverrides;
 //why separate from constructor: we create config manager,
 //then we create message manager, and _only then_ any
 //error messages may be given in an civilized manner
 //(and only then Die/AddLog can be used)...
 var
   ov, ovs, ovd, ovsd: string;
   j: integer;
 begin
   ovd:='config.ini';
   ovsd:='video audio controls';
   j:=0;
   While Ini[0].ReadString('config','file' + IntToStr(j), ovd) <> '' do
   begin
     ov:=Ini[0].ReadString('config','file' + IntToStr(j), ovd);
     ovs:=Ini[0].ReadString('config','file' + IntToStr(j) + '_override_sections_allowed', ovsd);
     if AttachIni(OptiPath(WorkingDir + ov) + ExtractFileName(ov)) then overlist.Last:=UpperCase(ovs);
     inc(j);
     ovd:='';
     ovsd:='';
   end;
 end;

 Procedure TConfigManager.Reload;
 begin
   Ini.Length:=1;
   //Ini.FreeContained;
   //Self.Length:=0;
   AttachOverrides;
 end;
 
 procedure TConfigManager.ScreamLimitsRead(fnum: integer; s,i: string; val, min, max: integer);
 begin
   Die(MI_ERROR_INVALID_CONFIG_PARAMETER,[Ini[fnum].FileName, s, i, val, min, max]);
 end;
